# -*- coding: utf-8 -*-
import json
import logging
from collections import OrderedDict
from datetime import datetime, timedelta
from itertools import chain


class Timer(object):
    def __init__(self, start_time=None):
        self.start_time = start_time or datetime.now()

    @property
    def elapsed(self):
        return datetime.now() - self.start_time

    def get_elapsed_seconds(self):
        return self.elapsed.total_seconds()

    def seconds_to_delta(self, delta):
        return (delta - self.elapsed).total_seconds()

    def seconds_to_shift(self, shift):
        return self.seconds_to_delta(timedelta(seconds=shift))


class Timeline(object):
    def __init__(self, logger, defaults=None, start_time=None):
        if isinstance(logger, basestring):
            logger = logging.getLogger(logger)
        self._logger = logger
        self._timer = Timer(start_time=start_time)
        self._id = str(self._timer.start_time.strftime('%d:%H%M%S.%f'))

    def event(self, event_name, extra=None):
        items = [
            ('ts', datetime.now().strftime('%Y-%m-%d %H:%M:%S')),
            ('tid', self._id),
            ('elapsed_ms', int(self._timer.elapsed.total_seconds() * 1000)),
            ('event', event_name),
        ]

        items = OrderedDict(chain(
            items,  # Чтобы установить значения в начало
            extra or [],
            items  # Чтобы не затёрлись случайно другими значениями
        ))

        message = self.make_format(items)

        self._logger.info(message)

    def make_format(self, items):
        return '\t'.join(
            '{}={}'.format(k, _somehow_to_str(v)) for k, v in items.items()
        )

    def with_defaults(self, defaults):
        return TimelineWithDefaults(self, defaults)


class TimelineJson(Timeline):
    def __init__(self, logger, defaults=None, start_time=None):
        super(TimelineJson, self).__init__(logger, defaults=defaults, start_time=start_time)

    def make_format(self, items):
        return json.dumps(items)


class TimelineWithDefaults(object):
    def __init__(self, timeline, defaults):
        self._timeline = timeline
        self._defaults = OrderedDict(defaults or []).items()

    def event(self, event_name, extra=None):
        self._timeline.event(
            event_name,
            self._defaults + OrderedDict(extra or []).items()
        )


def _somehow_to_str(value):
    try:
        return str(value)
    except Exception:
        return json.dumps(value, default=repr)
