# -*- coding: utf-8 -*-
import logging

from functools import wraps

from decorator import decorator

DATETIME_FORMAT = '%Y-%m-%d %H:%M:%S'
FALSE_VARIANTS = frozenset({'false', '0', 'null', 'none', ''})

log = logging.getLogger(__name__)


class Undefined(object):
    def __getattr__(self, name):
        return self

    def __getitem__(self, name):
        return self

    def __iter__(self):
        return iter([])

    def __nonzero__(self):
        return False

    def __json__(self):
        return None


undefined = Undefined()


def wrap(v):
    if isinstance(v, dict):
        return DictWrapper(v)

    if isinstance(v, list):
        return ListWrapper(v)

    return v


class DictWrapper(dict):
    def __getitem__(self, name):
        try:
            return wrap(dict.__getitem__(self, unicode(name)))

        except KeyError:
            return undefined

    def __getattr__(self, name):
        if name.startswith('__'):
            raise AttributeError

        return self[name]

    def __setattr__(self, name, value):
        self[name] = value


class ListWrapper(list):
    def __getitem__(self, i):
        try:
            return wrap(list.__getitem__(self, i))

        except IndexError:
            return undefined

    def __iter__(self):
        for v in list.__iter__(self):
            yield wrap(v)


class MergeableDict(dict):
    def __add__(self, other):
        assert isinstance(other, dict)
        return MergeableDict(self, **other)


def skip_None_values(d):
    if isinstance(d, dict):
        # Type(d) allow to work with dict subclasses and return same type
        return type(d)(
            filter(lambda (key, val): val is not None, d.iteritems())
        )

    return d


def dict_merge(*args):
    result = {}

    for d in args:
        assert isinstance(d, dict)
        result.update(d)

    return result


def str_to_bool(s):
    if isinstance(s, bool):
        return s
    elif isinstance(s, basestring) and s.lower() in FALSE_VARIANTS:
        return False
    elif s == 0 or s is None:
        return False
    return True


def pipe(sink):
    """Пропустит результат через функцию sink"""
    def deco(fn):
        @wraps(fn)
        def wrapper(*args, **kwargs):
            return sink(fn(*args, **kwargs))
        return wrapper
    return deco


@decorator
def profile_handler(fn, *args, **kwargs):
    import cProfile
    import pstats
    import StringIO

    pr = cProfile.Profile()
    pr.enable()

    result = fn(*args, **kwargs)

    pr.disable()
    s = StringIO.StringIO()
    sortby = 'cumulative'
    ps = pstats.Stats(pr, stream=s).sort_stats(sortby)
    ps.print_stats()
    print s.getvalue()

    return result
