# -*- coding: utf-8 -*-
from abc import abstractmethod, ABCMeta
from datetime import datetime, timedelta
from itertools import repeat

from travel.avia.ticket_daemon_api.jsonrpc.lib.caches import default_cache
from travel.avia.ticket_daemon_api.jsonrpc.lib.date import get_msk_now
from travel.avia.ticket_daemon_api.jsonrpc.models_utils.api_limits import jsend_api_limits


class AbstractAPILimitsStorage:
    __metaclass__ = ABCMeta

    @abstractmethod
    def set(self, *args, **kwargs):
        pass

    @abstractmethod
    def get(self, *args, **kwargs):
        pass


def update_period_seconds(update_period):
    """
    :type update_period: datetime.time|None
    :return int: Интервал обновления лимитов в секундах
    """
    if update_period is not None:
        return (update_period.hour * 60 + update_period.minute) * 60
    else:
        return 24 * 60 * 60


def limit_expired_seconds(update_time, update_period):
    """

    :return int: Количество секунд, через которое нужно сбросить счётчик лимитов
    """
    msk_now = get_msk_now()
    last_update_dt = datetime(
        *(list(msk_now.timetuple()[:3]) + [update_time.hour, update_time.minute]),
        tzinfo=msk_now.tzinfo
    )

    last_update_in_future = msk_now < last_update_dt
    if last_update_in_future:
        last_update_dt -= timedelta(days=1)

    current_day_delta = int((msk_now - last_update_dt).total_seconds())
    delta_s = update_period_seconds(update_period)
    return delta_s - (current_day_delta % delta_s) - 5


def LIMITS():
    return jsend_api_limits().get(InitSearchMCRLimits.ROUTE, {})


class InitSearchMCRLimits(AbstractAPILimitsStorage):
    ROUTE = 'init_search'

    @classmethod
    def set(cls, service, queried_partners):
        """

        :param Iterable queried_partners: список опрашиваемых партнеров
        :return dict: Оставшееся количество запросов по каждому партнёру
        """
        _limit = LIMITS().get(service)
        if _limit:
            usages = cls._get(service, queried_partners)
            updated_usages = dict(map(lambda (x, y): (x, y + 1), usages.iteritems()))

            store_time = limit_expired_seconds(_limit.update_time, _limit.update_period)
            default_cache.set_many(updated_usages, store_time)

            return cls.__format_limits(updated_usages, _limit.limit)

    @classmethod
    def get(cls, service, queried_partners):
        """

        :param Iterable queried_partners: список опрашиваемых партнеров
        :return dict: Оставшееся количество запросов по каждому партнёру
        """
        _limit = LIMITS().get(service)
        if _limit:
            usages = cls._get(service, queried_partners)
            return cls.__format_limits(usages, _limit.limit)

    @classmethod
    def bound(cls, service, limits):
        _limit = LIMITS().get(service)
        bound = {
            'next_update_time': limit_expired_seconds(_limit.update_time,
                                                      _limit.update_period)
        }
        for partner, left in limits.iteritems():
            bound[partner] = {'left': left, 'limit': _limit.limit}
        return bound

    @classmethod
    def _get(cls, service, queried_partners):
        """

        :return dict: Использованное количество запросов по каждому партнёру
        """
        keys = map(lambda p: cls.__format_key(service, p), queried_partners)
        usages = dict(zip(keys, repeat(0)))
        stored_usages = default_cache.get_many(keys)
        stored_usages = {key: int(limit) for key, limit in stored_usages.iteritems()}
        usages.update(stored_usages)
        return usages

    @classmethod
    def has_limits(cls, service):
        return service in LIMITS()

    @classmethod
    def __format_key(cls, service, partner):
        return '{};{};{};limit'.format(cls.ROUTE, service, partner)

    @classmethod
    def __partner_from_key(cls, partner_key):
        return partner_key.split(';')[-2]

    @classmethod
    def __format_limits(cls, usages, max_limit):
        """
        Преобразует словарь с числом использованных обращений к партнерам
         в словарь с числом оставшихся.
        """
        return {cls.__partner_from_key(key): max(0, max_limit - limit)
                for key, limit in usages.iteritems()}
