# -*- coding: utf-8 -*-
from collections import defaultdict
from itertools import imap

from django.conf import settings
from django.db.models import Q
from travel.avia.library.python.django_namedtuples.queryset import ModelInterface

from travel.avia.library.python.avia_data.models import (
    FlightReview, AviaCompany,
    FlightRating, FlightNumber
)
from travel.avia.library.python.avia_data.models.company import CompanyTariff
from travel.avia.library.python.common.models.schedule import AviaAlliance
from travel.avia.library.python.common.models.schedule import Company
from travel.avia.library.python.common.models.transport import TransportType

from travel.avia.library.python.ticket_daemon.memo import memoize, SimpleWarmGroup

company_warm_group = SimpleWarmGroup('avia-company-api')


@company_warm_group
@memoize()
def _aviacompanies_by_company_id():
    return {
        c.rasp_company_id: c
        for c in AviaCompany.objects.all().namedtuples(
            'rasp_company_id', 'cost_type', 'carryon_length', 'carryon_width',
            'carryon_height', 'carryon_dimensions_sum', 'baggage_rules',
            'baggage_rules_url', 'baggage_dimensions_sum',
            computational={
                'id': lambda values: values[0]
            })
    }


def get_avia_company_by_id(pk):
    return _aviacompanies_by_company_id().get(pk)


def get_aviacompanies_by_company_ids(ids):
    by_id = _aviacompanies_by_company_id()
    return [by_id[jd] for jd in ids if jd in by_id]


class CompanyInterface(ModelInterface):
    def get_title(self, lang):
        if lang == 'ru':
            return self.title_ru or self.title
        if lang == 'uk':
            return self.title_uk or self.get_title('ru')
        if lang == 'tr':
            return self.title_tr or self.get_title('en')
        if lang == 'en':
            return self.title_en

        return self.title_en

    def get_logo(self):
        return self.svg_logo2 or None

    def get_png_logo(self):
        return self.svg2png_logo2 or None


@company_warm_group
@memoize()
def _companies_by_id():
    langs = settings.ALLOW_LANGS
    fields = [
        'id', 'alliance_id', 'title', 'logo_bgcolor', 'svg_logo2', 'svg2png_logo2', 'url',
        'iata', 'sirena_id'
    ] + ['title_{}'.format(lang) for lang in langs]
    cc = Company.objects.filter(
        Q(t_type_id=TransportType.PLANE_ID) |
        Q(iata__isnull=False) |
        Q(sirena_id__isnull=False) |
        Q(icao__isnull=False)
    ).namedtuples(*fields, interface=CompanyInterface)

    return {c.id: c for c in cc}


def get_company_by_id(id):
    return _companies_by_id().get(id)


def get_companies_by_ids(ids):
    return filter(None, imap(get_company_by_id, ids))


@company_warm_group
@memoize()
def _alliances_by_id():
    cc = AviaAlliance.objects.all().namedtuples('id', 'title')

    return {c.id: c for c in cc}


def get_alliances_by_ids(ids):
    cc = _alliances_by_id()

    return [cc[jd] for jd in ids if jd in cc]


@company_warm_group
@memoize()
def _company_tariffs_by_id():
    cc = CompanyTariff.objects.all().namedtuples(
        'id', 'baggage_allowed', 'carryon', 'baggage_norm', 'carryon_norm',
        'published', 'avia_company_id', 'baggage_pieces'
    )

    return {c.id: c for c in cc}


@company_warm_group
@memoize()
def _company_tariffs_id_by_airline_id():
    return {
        ct.avia_company_id: ct.id
        for ct in _company_tariffs_by_id().itervalues()
    }


def get_company_tariffs_by_ids(ids):
    by_id = _company_tariffs_by_id()
    return [by_id[jd] for jd in ids if jd in by_id]


@company_warm_group
@memoize()
def _flight_ratings():
    return list(FlightRating.objects.all().namedtuples(
        'number', 'canceled', 'scores', 'delayed_less_30', 'delayed_30_60',
        'delayed_60_90', 'delayed_more_90'
    ))


def flight_ratings_by_numbers(numbers):
    rr = _flight_ratings()

    result = []
    for r in rr:
        if r.number in numbers:
            result.append(r)

    return result


@company_warm_group
@memoize()
def get_count_review_by_number():
    exited_reviews = set(
        FlightReview.objects.filter(
            enable_show=True
        ).values_list('id', flat=True)
    )
    nn = FlightNumber.objects.all().namedtuples('id', 'flight_number')

    number_to_review_relations = (
        FlightReview.flight_numbers
                    .through.objects.all()
                    .namedtuples('flightnumber_id', 'flightreview_id'))

    number_to_review = defaultdict(set)
    for relation in number_to_review_relations:
        if relation.flightreview_id in exited_reviews:
            number_to_review[relation.flightnumber_id].add(
                relation.flightreview_id
            )
    number_to_review = dict(number_to_review)

    result = defaultdict(int)
    for flight_number in nn:
        result[flight_number.flight_number] += len(number_to_review.get(
            flight_number.id, [])
        )
    result = dict(result)

    return dict(result)


def get_airline_tariff_id(airline_id):
    return _company_tariffs_id_by_airline_id().get(airline_id)
