# -*- coding: utf-8 -*-
import logging
from collections import defaultdict, OrderedDict
from itertools import chain, imap

import pytz
from django.conf import settings
from django.db.models import Q
from travel.avia.library.python.django_namedtuples.queryset import ModelInterface

from travel.avia.library.python.common.models.geo import (
    Country, Station, Station2Settlement, Settlement, StationType, StationCode,
)
from travel.avia.library.python.common.models.transport import TransportType, TransportModel

from travel.avia.ticket_daemon_api.jsonrpc.lib.date import get_msk_now, get_utc_now
from travel.avia.library.python.ticket_daemon.memo import memoize, SimpleWarmGroup
from translations import (
    get_title, get_from_title, get_in_title, get_to_title, ACCUSATIVE_CASE
)

logger = logging.getLogger(__name__)
geo_warm_group = SimpleWarmGroup('geo-api')


@geo_warm_group
@memoize()
def _settlement_ids_by_airport_id():
    by_airport = defaultdict(OrderedSet)

    # Сначала города, привязанные явно
    for airport in _airports_by_id().values():
        if airport.settlement_id:
            by_airport[airport.id].add(airport.settlement_id)

    # После — через station2settlement
    for s2s in _station2settlement():
        by_airport[s2s.station_id].add(s2s.settlement_id)

    return {
        settlement_id: tuple(stations_ids)
        for settlement_id, stations_ids in by_airport.iteritems()
    }


def get_settlement_ids_by_airport_id(settlement_id):
    return _settlement_ids_by_airport_id().get(settlement_id)


class PointInterface(ModelInterface):
    @property
    def type(self):
        if self.point_key.startswith('c'):
            return Settlement
        elif self.point_key.startswith('s'):
            return Station

    def get_title(self, lang):
        return get_title(self.new_L_title_id, lang)

    def get_from_title(self, lang):
        return get_from_title(self.new_L_title_id, lang)

    def get_in_title(self, lang):
        return get_in_title(self.new_L_title_id, lang)

    def get_to_title(self, lang):
        return get_to_title(self.new_L_title_id, lang)

    def get_locative_title(self, lang):
        return get_in_title(self.new_L_title_id, lang)

    def get_genitive_title(self, lang):
        return get_from_title(self.new_L_title_id, lang)

    def get_accusative_title(self, lang):
        return get_title(self.new_L_title_id, lang, ACCUSATIVE_CASE)

    def get_prefix(self, lang):
        if lang == 'ru':
            return self.prefix_ru or self.prefix
        if lang == 'uk':
            return self.prefix_uk or self.get_prefix('ru')
        if lang == 'tr':
            return self.prefix_tr or self.get_prefix('en')
        if lang == 'en':
            return self.prefix_en

    @property
    def local_time(self):
        if self.time_zone:
            try:
                return get_utc_now().astimezone(pytz.timezone(self.time_zone))
            except pytz.UnknownTimeZoneError:
                logger.exception('Unknown pytz TimeZone %s', self.time_zone)
                raise ValueError('Unknown pytz TimeZone %s' % self.time_zone)
        return get_msk_now()

    def get_related_settlement(self):
        if issubclass(self.type, Settlement):
            return self
        if issubclass(self.type, Station):
            return get_settlement_by_id(
                self.settlement_id or
                next(iter(get_settlement_ids_by_airport_id(self.id) or []), None)
            )
        raise RuntimeError()

    def get_related_settlement_ids(self):
        if issubclass(self.type, Settlement):
            ids = {self.id}
            for airport_id in get_airport_ids_by_settlement_id_except_hidden().get(self.id, []):
                ids.update(get_settlement_ids_by_airport_id(airport_id))
            return ids
        if issubclass(self.type, Station):
            ids = get_settlement_ids_by_airport_id(self.id)
            return set(list(ids)) if ids else set()
        raise RuntimeError()

    def get_airports_except_hidden(self):
        if issubclass(self.type, Settlement):
            airport_ids_by_settlement_id = get_airport_ids_by_settlement_id_except_hidden()
            return list(airport_ids_by_settlement_id.get(self.id, []))
        if issubclass(self.type, Station):
            return [self.id]
        return []


@geo_warm_group
@memoize()
def _country_by_id():
    fields = [
        'id', 'code',
    ]
    cc = list(Country.objects.all().namedtuples(
        *fields,
        computational={
            'point_key': (lambda row: 'l{}'.format(row[0])),
        },
        interface=PointInterface)
    )

    return {c.id: c for c in cc}


def get_country_by_id(id_):
    return _country_by_id().get(id_)


@geo_warm_group
@memoize()
def _settlement_by_id():
    # todo можно выкачивать только города с аэропортами

    fields = [
        'id', 'country_id', 'iata', 'sirena_id',
        'new_L_title_id', 'title_ru_preposition_v_vo_na', 'time_zone'
    ]
    cc = list(Settlement.objects.all()
              .namedtuples(*fields,
                           computational={
                               'point_key': (lambda row: 'c{}'.format(row[0])),
                               'settlement_id': (lambda row: row[0]),
                           },
                           interface=PointInterface))

    return {c.id: c for c in cc}


def get_settlement_by_id(id):
    return _settlement_by_id().get(id)


def get_settlements_by_ids(ids):
    return filter(None, imap(get_settlement_by_id, ids))


@geo_warm_group
@memoize()
def _settlement_by_iata():
    return {s.iata: s for s in _settlement_by_id().itervalues()}


@geo_warm_group
@memoize()
def _settlement_by_sirena():
    return {s.sirena_id: s for s in _settlement_by_id().itervalues()}


@geo_warm_group
@memoize()
def _airports_by_id():
    """
    :rtype: dict[int, Station]
    """
    fields = [
        'id',
        'settlement_id',
        'country_id',
        'time_zone',
        'sirena_id',
        'hidden',
        't_type_id', 'station_type_id',
        'new_L_title_id', 'title_ru_preposition_v_vo_na',
    ]
    station_ids = set(chain(
        _station_id_by_iata().itervalues(),
        _station_id_by_sirena().itervalues()
    ))
    iata_by_station_id = _stations_iatas_by_station_id()

    qs = Q(id__in=station_ids) | Q(t_type_id=TransportType.PLANE_ID)

    cc = list(Station.objects.filter(qs).namedtuples(
        *fields,
        computational={
            'point_key': (lambda row: 's{}'.format(row[0])),
            'iata': (lambda row: iata_by_station_id.get(row[0])),
        },
        interface=PointInterface
    ))

    return {c.id: c for c in cc}


def get_station_by_id(id):
    return _airports_by_id().get(id)


def get_stations_by_ids(ids):
    return filter(None, imap(get_station_by_id, ids))


@geo_warm_group
@memoize()
def _station2settlement():
    return list(
        Station2Settlement.objects
            .filter(station_id__in=_airports_by_id().keys())
            .namedtuples('station_id', 'settlement_id')
    )


class OrderedSet(OrderedDict):
    """
    В итоге чтобы получить значения, нужно будет преобразовать в список
    """

    def add(self, value):
        if value not in self:
            self[value] = None


@geo_warm_group
@memoize()
def get_airport_ids_by_settlement_id():
    by_settlement = defaultdict(OrderedSet)

    for airport in _airports_by_id().values():
        if airport.settlement_id:
            by_settlement[airport.settlement_id].add(airport.id)

    for s2s in _station2settlement():
        by_settlement[s2s.settlement_id].add(s2s.station_id)

    return {
        settlement_id: tuple(stations_ids)
        for settlement_id, stations_ids in by_settlement.iteritems()
    }


@geo_warm_group
@memoize()
def get_airport_ids_by_settlement_id_except_hidden():
    by_settlement = defaultdict(OrderedSet)

    hidden_airports = set()
    for airport in _airports_by_id().values():
        if airport.hidden:
            hidden_airports.add(airport.id)
            continue
        if airport.settlement_id:
            by_settlement[airport.settlement_id].add(airport.id)

    for s2s in _station2settlement():
        if s2s.station_id not in hidden_airports:
            by_settlement[s2s.settlement_id].add(s2s.station_id)

    return {
        settlement_id: tuple(stations_ids)
        for settlement_id, stations_ids in by_settlement.iteritems()
    }


class StationTypeInterface(ModelInterface):
    def get_title(self, lang):
        if lang == 'ru':
            return self.name_ru
        if lang == 'uk':
            return self.name_uk or self.get_name('ru')
        if lang == 'tr':
            return self.name_tr or self.get_name('en')
        if lang == 'en':
            return self.name_en

        return ''

    def get_prefix(self, lang):
        if lang == 'ru':
            return self.prefix_ru
        if lang == 'uk':
            return self.prefix_uk or self.get_prefix('ru')
        if lang == 'tr':
            return self.prefix_tr or self.get_prefix('en')
        if lang == 'en':
            return self.prefix_en

        return ''


@geo_warm_group
@memoize()
def _station_types_by_id():
    langs = settings.ALLOW_LANGS
    fields = ['id'] + ['name_{}'.format(lang) for lang in langs] + ['prefix_{}'.format(lang) for lang in langs]
    cc = list(StationType.objects.all().namedtuples(*fields, interface=StationTypeInterface))

    return {c.id: c for c in cc}


def get_stations_type_by_id(pk):
    return _station_types_by_id()[pk]


@geo_warm_group
@memoize()
def _stations_iatas_by_station_id():
    return dict(StationCode.objects.filter(system__code='iata')
                .values_list('station_id', 'code'))


@geo_warm_group
@memoize()
def _station_id_by_iata():
    return {
        iata: sid for sid, iata in _stations_iatas_by_station_id().iteritems()
    }


def station_iata(station):
    return _stations_iatas_by_station_id().get(station.id)


@geo_warm_group
@memoize()
def _stations_sirenas_by_station_id():
    return dict(StationCode.objects.filter(system__code='sirena')
                .values_list('station_id', 'code'))


@geo_warm_group
@memoize()
def _station_id_by_sirena():
    return {
        sirena: sid
        for sid, sirena in _stations_sirenas_by_station_id().iteritems()
    }


@memoize(lambda system_code: system_code)
def _station_id_by_codesystem(system_code):
    return dict(StationCode.objects.filter(system__code=system_code)
                .values_list('code', 'station_id'))


@memoize(lambda code: code)
def get_station_by_iata_or_sirena(code):
    station_id = (_station_id_by_iata().get(code) or
                  _station_id_by_sirena().get(code))
    if station_id:
        return _airports_by_id().get(station_id)


@memoize()
def _point_tuple_by_key():
    points = {
        's{}'.format(s.id): s for s in _airports_by_id().values()
    }
    points.update({
        'c{}'.format(c.id): c
        for c in _settlement_by_id().values()
    })
    return points


def get_point_tuple_by_key(key):
    return _point_tuple_by_key().get(key)


def get_point_by_iata_or_sirena(code):
    return (
        _settlement_by_iata().get(code) or
        _settlement_by_sirena().get(code) or
        get_station_by_iata_or_sirena(code)
    )


class TransportModelInterface(ModelInterface):
    def get_title(self, lang):
        if lang == 'en':
            return self.title_en or self.title

        return self.title


@geo_warm_group
@memoize()
def get_transort_models_by_id():
    tt = TransportModel.objects.all().namedtuples(
        'id', 'title', 'title_en',
        interface=TransportModelInterface
    )

    return {t.id: t for t in tt}


@SimpleWarmGroup('geo')
@memoize()
def transport_type_codes():
    return dict(TransportType.objects.all().values_list('id', 'code'))
