# -*- coding: utf-8 -*-
from travel.avia.library.python.django_namedtuples.queryset import ModelInterface

from travel.avia.library.python.common.utils.text import title_with_preposition
from travel.avia.library.python.common.models.geo import Station, Settlement
from travel.avia.library.python.common.models.transport import TransportType

from travel.avia.library.python.ticket_daemon.memo import memoize, SimpleWarmGroup
from translations import (
    get_title, get_from_title, get_in_title, get_to_title, ACCUSATIVE_CASE
)


geo_rasp_warm_group = SimpleWarmGroup('geo-rasp-api')


class PointInterface(ModelInterface):
    def get_title(self, lang):
        return get_title(self.new_L_title_id, lang)

    def get_from_title(self, lang):
        return title_with_preposition(u'из', get_from_title(self.new_L_title_id, lang))

    def get_in_title(self, lang):
        return title_with_preposition(
            self.title_ru_preposition_v_vo_na or u'в',
            get_in_title(self.new_L_title_id, lang)
        )

    def get_to_title(self, lang):
        return title_with_preposition(
            self.title_ru_preposition_v_vo_na or u'в',
            get_to_title(self.new_L_title_id, lang)
        )

    def get_locative_title(self, lang):
        return get_in_title(self.new_L_title_id, lang)

    def get_genitive_title(self, lang):
        return get_from_title(self.new_L_title_id, lang)

    def get_accusative_title(self, lang):
        return get_title(self.new_L_title_id, lang, ACCUSATIVE_CASE)

    def get_prefix(self, lang):
        if lang == 'ru':
            return self.prefix_ru or self.prefix
        if lang == 'uk':
            return self.prefix_uk or self.get_prefix('ru')
        if lang == 'tr':
            return self.prefix_tr or self.get_prefix('en')
        if lang == 'en':
            return self.prefix_en


@geo_rasp_warm_group
@memoize()
def _settlement_by_id():
    # todo можно выкачивать только города с аэропортами

    fields = [
        'id', 'country_id', 'iata', 'sirena_id',
        'new_L_title_id', 'title_ru_preposition_v_vo_na'
    ]
    cc = list(Settlement.objects.all()
              .namedtuples(*fields,
                           computational={
                               'point_key': (lambda row: 'c{}'.format(row[0])),
                               'settlement_id': (lambda row: row[0]),
                           },
                           interface=PointInterface))

    return {
        c.id: c
        for c in cc
        }


def get_settlements_by_ids(ids):
    by_id = _settlement_by_id()
    return [by_id[jd] for jd in ids if jd in by_id]


@geo_rasp_warm_group
@memoize()
def _stations_by_id():
    fields = [
        'id', 't_type_id', 'station_type_id', 'settlement_id', 'sirena_id',
        'new_L_title_id', 'title_ru_preposition_v_vo_na'
    ]
    cc = list(Station.objects.filter(t_type_id=TransportType.PLANE_ID)
              .namedtuples(*fields,
                           computational={
                               'point_key': (lambda row: 's{}'.format(row[0]))
                           },
                           interface=PointInterface))

    return {
        c.id: c
        for c in cc
        }


def get_stations_by_ids(ids):
    by_id = _stations_by_id()
    return [by_id[jd] for jd in ids if jd in by_id]
