# -*- coding: utf-8 -*-
import itertools
import logging

from django.conf import settings
from travel.avia.library.python.django_namedtuples.queryset import ModelInterface

from travel.avia.library.python.common.models.partner import DohopVendor, Partner
from travel.avia.library.python.avia_data.models.amadeus_merchant import AmadeusMerchant

from travel.avia.library.python.ticket_daemon.memo import memoize, SimpleWarmGroup


logger = logging.getLogger(__name__)
order_warm_group = SimpleWarmGroup('order-api')


class PartnerInterface(ModelInterface):
    _url_postfix = '/orig'

    def get_title(self, lang, national_version):
        return (
            self._get_national_title(lang, national_version) or
            self._get_title(lang) or
            self.title
        )

    def _get_national_title(self, lang, national_version):
        if lang == 'uk':
            return (
                self._get_exact_national_title(lang, national_version) or
                self._get_exact_national_title('ru', national_version)
            )
        if lang == 'tr':
            return (
                self._get_exact_national_title(lang, national_version) or
                self._get_exact_national_title('en', national_version)
            )
        return self._get_exact_national_title(lang, national_version)

    def _get_exact_national_title(self, lang, national_version):
        field = 'national_{}_title_{}'.format(national_version, lang)
        if not hasattr(self, field):
            field = 'national_{}_title_{}'.format(national_version, 'en')
        return getattr(self, field)

    def _get_title(self, lang):
        if lang == 'uk':
            return self._get_exact_title(lang) or self._get_exact_title('ru')
        if lang == 'tr':
            return self._get_exact_title(lang) or self._get_exact_title('en')
        return self._get_exact_title(lang)

    def _get_exact_title(self, lang):
        field = 'title_{}'.format(lang)
        if not hasattr(self, field):
            field = 'title_{}'.format('en')
        return getattr(self, field)

    def get_logo(self, national_version):
        field_name = 'logo2_svg_{}'.format(national_version)
        logo = getattr(self, field_name, None)
        if logo:
            return logo
        if national_version != 'ru' and self.logo2_svg_ru:
            return self.logo2_svg_ru

    def get_png_logo(self, national_version):
        field_name = 'logo2_svg2png_{}'.format(national_version)
        logo = getattr(self, field_name, None)
        if logo:
            return self._append_url_postfix_for_png(logo)
        if national_version != 'ru' and self.logo2_svg2png_ru:
            return self._append_url_postfix_for_png(self.logo2_svg2png_ru)

    def _append_url_postfix_for_png(self, logo):
        if self._url_postfix and not logo.endswith((self._url_postfix, '.png')):
            return logo + self._url_postfix

    def get_variant_cache_ttl(self):
        if hasattr(self, 'variant_cache_ttl'):
            return getattr(self, 'variant_cache_ttl')
        return getattr(self, 'dohop_cache_ttl')


def _get_partners_list():
    langs = settings.ALLOW_LANGS
    national_versions = settings.ALLOW_NATIONAL_VERSIONS

    common_fields = ['id', 'code', 'is_aviacompany', 'title', 'site_url'] + [
        'title_{}'.format(lang) for lang in langs
    ] + [
        'national_{}_title_{}'.format(nv, lang)
        for nv, lang in itertools.product(national_versions, langs)
    ] + [
        'logo2_svg_{}'.format(nv) for nv in national_versions
    ] + [
        'logo2_svg2png_{}'.format(nv) for nv in national_versions
    ]

    partner_fields = common_fields + ['pricing_model', 'variant_cache_ttl']
    common_partners = Partner.objects.all().namedtuples(
        *partner_fields,
        interface=PartnerInterface
    )

    dohop_fields = common_fields + ['dohop_cache_ttl']
    dohop_partners = DohopVendor.objects.all().namedtuples(
        *dohop_fields,
        computational={
            'pricing_model': lambda d: DohopVendor.pricing_model,
        },
        interface=PartnerInterface
    )

    amadeus_fields = common_fields + ['variant_cache_ttl']
    amadeus_partners = AmadeusMerchant.objects.all().namedtuples(
        *amadeus_fields,
        computational={
            'pricing_model': lambda d: AmadeusMerchant.pricing_model,
        },
        interface=PartnerInterface
    )
    return list(itertools.chain(common_partners, dohop_partners, amadeus_partners))


@order_warm_group
@memoize()
def get_partners_by_code():
    return {p.code: p for p in _get_partners_list()}


def _get_partner_by_code(code):
    return get_partners_by_code().get(code)


def get_partner_by_code(code):
    partner = _get_partner_by_code(code)
    if partner is None:
        logger.error('Partner code %s is not in cache', code)
        get_partners_by_code.reset()
        partner = _get_partner_by_code(code)
    return partner
