# -*- coding: utf-8 -*-
from collections import defaultdict

from enum import Enum

from travel.avia.library.python.ticket_daemon.memo import SimpleWarmGroup, memoize
from travel.avia.ticket_daemon_api.jsonrpc.models_utils.geo import _settlement_by_id, _station2settlement, _airports_by_id

search_codes_warmup_group = SimpleWarmGroup('search_codes')


def have_any_code(point, code_systems):
    """
    :param point: Точка, для которой нужнл проверить наличие кода хотя бы в одной из кодовой систем
    :param code_systems: Кодовые системы обозначения точек. Сейчас валидны iata и sirena_id
    :type code_systems: typing.Iterable[str]

    """
    return any(getattr(point, code_system, None) is not None for code_system in code_systems)


class PointReason(Enum):
    STATION_HAS_CODE_FOR_STATION_SEARCH = 1
    STATION_HAS_NO_CODE_FOR_STATION_SEARCH = 2
    SETTLEMENT_HAS_CODE_FOR_SETTLEMENT_SEARCH = 3
    SETTLEMENT_HAS_NO_CODE_FOR_SETTLEMENT_SEARCH = 4
    SETTLEMENT_S2S_SEARCH = 5


def _search_point_keys_by_point_key(code_systems):
    """
    :param code_systems: Кодовые системы обозначения точек. Сейчас валидны iata и sirena_id
    :type code_systems: typing.Iterable[str]

    На выходе получаем маппинг входящей точки (с23, s65036) к списку привязанных к ней
    напрямую или косвенно точек для поиска (возможно включая саму эту точку) [c15, c40, s3216, c55]
    при наличии у них кода в системах code_systems
    аэропорт ->
        first(ключ "аэропорт" || ключ "город аэропорта")
    город ->
        first(точка "город", если в нём есть аэропорты и у города есть код в кодовой системе code_systems ||
              список точек "аэропорт" города при наличии кода в кодовой системе code_systems) +

        список(
            first(
                точка s2s "аэропорт" при наличии у аэропорта кода в кодовой системе code_systems ||
                точка "город" s2s аэропорта, при наличии у города соответствующего кода в кодовой системе code_systems
            ) ,
        )

    :rtype: dict[unicode, list[unicode]]
    """

    settlement_by_id = _settlement_by_id()
    station_by_id = _airports_by_id()
    stations_by_settlement_id = defaultdict(list)

    for station in station_by_id.itervalues():
        if (
            station.settlement_id
        ):
            stations_by_settlement_id[station.settlement_id].append(station)

    s2s_stations_by_settlement_id = defaultdict(set)
    for s2s in _station2settlement():
        station = station_by_id.get(s2s.station_id)
        if station:
            s2s_stations_by_settlement_id[s2s.settlement_id].add(station)

    def get_station_search_points(station_id):
        """
        Generates list of search points with reasons
        :rtype: typing.Generator[typing.Tuple[travel.avia.library.python.common.models_utils.geo.Point, typing.Iterable[PointReason]], None, None]
        """
        _station = station_by_id.get(station_id)
        if have_any_code(_station, code_systems):
            yield _station, (PointReason.STATION_HAS_CODE_FOR_STATION_SEARCH,)
        elif _station.settlement_id:
            s2s_settlement = settlement_by_id.get(_station.settlement_id)
            if s2s_settlement and have_any_code(s2s_settlement, code_systems):
                yield s2s_settlement, (PointReason.STATION_HAS_NO_CODE_FOR_STATION_SEARCH,)

    def get_settlement_related_points_by_settlement_id(settlement_id):
        """
        Generates list of search points with reasons
        :rtype: typing.Generator[typing.Tuple[travel.avia.library.python.common.models_utils.geo.Point, int], None, None]
        """
        settlement_ = settlement_by_id.get(settlement_id)
        if settlement_:
            settlement_stations = stations_by_settlement_id.get(settlement_id, [])
            if have_any_code(settlement_, code_systems):
                yield settlement_, (PointReason.SETTLEMENT_HAS_CODE_FOR_SETTLEMENT_SEARCH,)
            else:
                for settlement_station in settlement_stations:
                    if have_any_code(settlement_station, code_systems):
                        yield settlement_station, (PointReason.SETTLEMENT_HAS_NO_CODE_FOR_SETTLEMENT_SEARCH,)
            for s2s_station in s2s_stations_by_settlement_id.get(settlement_id, []):
                for related_point, point_reasons in get_station_search_points(s2s_station.id):
                    yield related_point, (point_reasons + (PointReason.SETTLEMENT_S2S_SEARCH,))

    by_point_key = defaultdict(list)

    for station in station_by_id.itervalues():
        pkey = station.point_key
        for point, reasons in get_station_search_points(station.id):
            if point and point.point_key not in by_point_key[pkey]:
                by_point_key[pkey].append((point.point_key, reasons))

    for settlement in settlement_by_id.itervalues():
        pkey = settlement.point_key
        for point, reasons in get_settlement_related_points_by_settlement_id(settlement.id):
            if point and point.point_key not in by_point_key[pkey]:
                by_point_key[pkey].append((point.point_key, reasons))

    return by_point_key


@search_codes_warmup_group
@memoize()
def _point_keys_for_search_by_point_key():
    """
    :rtype: dict[unicode, list[unicode]]
    """
    return _search_point_keys_by_point_key(code_systems=('iata', 'sirena_id'))


def get_codes_for_search(point_key):
    """
    :param unicode point_key:
    :return: iterator over iata codes
    :rtype: typing.Iterable[unicode]
    """

    return _point_keys_for_search_by_point_key().get(point_key)
