# -*- coding: utf-8 -*-
from django.conf import settings

from travel.avia.library.python.common.models.translations import TranslatedTitle, TranslatedText
from travel.avia.library.python.ticket_daemon.memo import SimpleWarmGroup, memoize

translate_warm_group = SimpleWarmGroup('translate')

TRANSLATIONS_ALLOW_LANGS = settings.TRANSLATIONS_ALLOW_LANGS


def point_title_ids():
    from .geo import _settlement_by_id, _airports_by_id
    ids = set(s.new_L_title_id for s in _settlement_by_id().values())
    ids |= set(s.new_L_title_id for s in _airports_by_id().values())
    return ids


@translate_warm_group
@memoize()
def _translate_title_by_id():
    fields = [
        'id',
        'ru_genitive',
        'ru_dative',
        'ru_accusative',
        'ru_locative',

        'uk_accusative'
    ] + ['{}_nominative'.format(lang) for lang in TRANSLATIONS_ALLOW_LANGS]
    ids = point_title_ids()
    cc = list(TranslatedTitle.objects.filter(id__in=ids).namedtuples(*fields))

    return {
        c.id: c
        for c in cc
    }


@translate_warm_group
@memoize()
def _translate_text_by_id():
    fields = [
        'id',
    ] + TRANSLATIONS_ALLOW_LANGS
    # Сейчас нигде не используются
    # и можно не кэшировать ids = point_title_ids()
    ids = []
    cc = list(TranslatedText.objects.filter(id__in=ids).namedtuples(*fields))

    return {
        c.id: c
        for c in cc
    }

NOMINATIVE_CASE = 'nominative'
GENITIVE_CASE = 'genitive'
DATIVE_CASE = 'dative'
ACCUSATIVE_CASE = 'accusative'
LOCATIVE_CASE = 'locative'

ALLOWED_CASES = {
    NOMINATIVE_CASE, GENITIVE_CASE, DATIVE_CASE,
    ACCUSATIVE_CASE, LOCATIVE_CASE
}
FALLBACKS = {
    'ru': None,
    'en': 'ru',
    'tr': 'en',
    'de': 'en',
    'uk': 'ru'
}


def get_title_translate_by_id(pk, lang, case):
    assert case in ALLOWED_CASES
    assert lang in TRANSLATIONS_ALLOW_LANGS

    translates = _translate_title_by_id()
    translate = translates.get(pk, None)
    if not translate:
        return

    return _get_title_translate_value_with_fallbacks(translate, lang, case)


def get_title(translation_id, lang, case=NOMINATIVE_CASE):
    if lang != 'ru':
        case = NOMINATIVE_CASE

    return get_title_translate_by_id(translation_id, lang, case)


def get_from_title(translation_id, lang):
    return get_title(translation_id, lang, GENITIVE_CASE)


def get_in_title(translation_id, lang):
    return get_title(translation_id, lang, LOCATIVE_CASE)


def get_to_title(translation_id, lang):
    return get_title(translation_id, lang, ACCUSATIVE_CASE)


def _get_title_translate_value_with_fallbacks(title_translate, lang, case):
    def format_key(l, c):
        return '{}_{}'.format(l, c)
    value = getattr(title_translate, format_key(lang, case), None)

    if not value and case != NOMINATIVE_CASE:
        value = _get_title_translate_value_with_fallbacks(
            title_translate, lang, NOMINATIVE_CASE
        )

    if not value and FALLBACKS[lang]:
        value = getattr(
            title_translate,
            format_key(FALLBACKS[lang], case), None
        )

    return value
