# -*- coding: utf-8 -*-
import logging
import operator
from datetime import datetime

from django.conf import settings

from travel.avia.library.python.avia_data.libs.currency.rates.caching import CurrencyRatesCache
from travel.avia.library.python.common.utils.currency_converter import ConverterCache

from travel.avia.ticket_daemon_api.jsonrpc.lib.date import DateTimeDeserializer
from travel.avia.ticket_daemon_api.jsonrpc.lib.http import update_query_string
from travel.avia.ticket_daemon_api.jsonrpc.lib.utils import dict_merge
from travel.avia.ticket_daemon_api.jsonrpc.models_utils.geo_rasp import (
    get_stations_by_ids, get_settlements_by_ids
)
from travel.avia.ticket_daemon_api.jsonrpc.models_utils.order import get_partner_by_code
from travel.avia.ticket_daemon_api.jsonrpc.complete_results import avia_deep_link, avia_order_link

log = logging.getLogger(__name__)
RASP_UNKNOWN_UTM_CAMPAIGNS = 'unknown'
RASP_UTM_CAMPAIGNS = {
    'rasp_api_public': 'api',
    'rasp_mobile': 'mobile',
    'rasp_morda': 'desktop',
    'rasp_morda_backend': 'desktop',
    'rasp_touch': 'touch',
}


def _utm_marks(query, utm_medium, utm_campaign=None):
    _utm_campaign = utm_campaign or RASP_UTM_CAMPAIGNS.get(
        query.service,
        RASP_UNKNOWN_UTM_CAMPAIGNS
    )

    if _utm_campaign == RASP_UNKNOWN_UTM_CAMPAIGNS:
        log.warning('Unknown service: %s', query.service)

    return {
        'utm_source': 'rasp',
        'utm_medium': utm_medium,
        'utm_campaign': _utm_campaign,
        'utm_content': query.national_version
    }


def _load_currency_rates(national_version):
    rates_caches = filter(None, [
        ConverterCache.load(national_version),
        CurrencyRatesCache.load(national_version),
    ])
    if not rates_caches:
        return None
    return dict_merge(*[cache.rates for cache in rates_caches])


def _avia_deep_link(query, v):
    url = avia_deep_link(query, v, clid=2289669)
    return update_query_string(url, _utm_marks(query, 'redirect'))


def _avia_order_link(query, v):
    url = avia_order_link(query, v, clid=2289669)
    return update_query_string(
        url,
        _utm_marks(query, utm_medium='search_result', utm_campaign='order_link')
    )


def _key2(routes, flights):
    return u','.join(
        u'{}.{}'.format(
            flights[route]['number'],
            flights[route]['departure'].get('local', '')[:-3]
        ) for route in routes
    )


def _itinerary_segment_keys(segment):
    number = segment['number'].encode('utf-8')

    return ['{}.{}'.format(
        number,
        segment['departure']['local'].strftime(f) if segment[
            'departure'] else '',
    ) for f in ['%m%dT%H%M', '%Y-%m-%dT%H:%M']]


def _itinerary_keys(segments):
    itinerary_segments_old = []
    itinerary_segments_new = []

    for s in segments:
        old_key, new_key = _itinerary_segment_keys(s)
        itinerary_segments_old.append(old_key)
        itinerary_segments_new.append(new_key)

    return '_'.join(itinerary_segments_old), ','.join(itinerary_segments_new)


def _segment_key(s):
    return '-'.join(map(str, (
        s['number'].encode('utf-8'),
        s['company'],
        s['station_from'],
        s['station_to'],
        s['departure']['local'].strftime('%m%d%H%M') if s['departure'] else None,
        s['arrival']['local'].strftime('%m%d%H%M') if s['arrival'] else None
    )))


def _reformat_date(dt_info, deserializer):
    dt_info['local'] = deserializer.deserialize(dt_info['local'])


def rasp_complete_results(query, search_result):
    """

    :type query: jsonrpc.query.Query
    :type search_result: typing.Dict[str, jsonrpc.lib.result.collector.variants_fabric.ApiVariants]
    :rtype: dict
    """
    rates = _load_currency_rates(query.national_version)
    currency = settings.AVIA_NATIONAL_CURRENCIES.get(query.national_version)

    variants = []
    partner_codes = set()

    for p_code, p_variants in search_result.iteritems():
        p_flights = dict(p_variants.flights)
        partner = get_partner_by_code(p_code)
        for fare in p_variants.variants:
            fare['national_tariff'] = _get_national_tariff(
                value=fare['tariff']['value'],
                from_currency=fare['tariff']['currency'],
                to_currency=currency, rates=rates
            )
            if not fare['national_tariff']:
                continue

            fare['partner_code'] = p_code
            partner_codes.add(p_code)

            if p_code in settings.CHARTER_PARTNERS:
                fare['charter'] = True

            fare['from_company'] = partner and partner.is_aviacompany or False
            fare['query_time'] = p_variants.query_time

            fare['forward'] = {
                'key2': _key2(fare['route'][0], p_flights)
            }
            fare['backward'] = {
                'key2': _key2(fare['route'][1], p_flights)
            }

            fare['deep_link'] = _avia_deep_link(query, fare)
            fare['order_link'] = _avia_order_link(query, fare)  # Todo: Лишнее, т.к. перестало использоваться

            variants.append(fare)

    expired_date = datetime.utcfromtimestamp(
        min(variants, key=operator.itemgetter('expire'))['expire']
    ) if variants else None

    reference = {
        'flights': {},
        'itineraries': {},
        'stations': set(),
        'settlements': set(),
        'partners': map(get_partner_by_code, partner_codes),
    }

    deserializer = DateTimeDeserializer()

    flights = {
        route: flight
        for s in search_result.itervalues()
        for route, flight in s.flights.iteritems()
    }
    for flight in flights.itervalues():
        flight['station_from'] = flight['from']
        flight['station_to'] = flight['to']
        _reformat_date(flight['arrival'], deserializer)
        _reformat_date(flight['departure'], deserializer)
        flight['new_key'] = flight['key']
        flight['key'] = _segment_key(flight)  # Перетираем ключи нового формата староформатными
        _fill_flight(flight, reference)
    reference['flights'] = reference['flights'].values()

    for v in variants:
        fwd_segments = [flights[route] for route in v['route'][0]]
        key, key2 = _itinerary_keys(fwd_segments)
        reference['itineraries'][key] = [flight['key'] for flight in fwd_segments]
        v['forward_itinerary'] = key

        bwd_segments = [flights[route] for route in v['route'][1]]
        key, key2 = _itinerary_keys(bwd_segments)
        reference['itineraries'][key] = [flight['key'] for flight in bwd_segments]
        v['backward_itinerary'] = key

    reference['stations'] = get_stations_by_ids(reference['stations'])
    reference['settlements'] = get_settlements_by_ids({
        s.settlement_id for s in reference['stations'] if s.settlement_id
    })

    return {
        'reference': reference,
        'variants': variants,
        'expired_date': expired_date,
    }


def _get_national_tariff(value, from_currency, to_currency, rates):
    if from_currency == to_currency:
        return {
            'value': value,
            'currency': from_currency,
            'base_value': value,
        }

    if not rates:
        return None

    try:
        if from_currency not in rates:
            raise ValueError(
                'Tariff currency [%s] not in rates', from_currency)

        if to_currency not in rates:
            raise ValueError('Target currency [%s] not in rates', to_currency)
        converted_value = float(value) * rates[from_currency] / rates[to_currency]

        return {
            'value': converted_value,
            'currency': to_currency,
            'base_value': converted_value,
        }

    except (ValueError, KeyError) as exc:
        log.warning(
            "Couldn't convert tariff to currency [%s]: %s", to_currency, exc)
        return None


def _fill_flight(flight, ref):
    key = flight['key']
    if key not in ref['flights']:
        ref['flights'][key] = {
            'key': key,
            'number': flight['number'],
            't_type_code': 'plane',
            'company': flight['company'],
            'station_from': flight['station_from'],
            'station_to': flight['station_to'],
            'departure': flight['departure'],
            'arrival': flight['arrival'],

            # Поставил заглушки-значения чтобы в расписаниях не сломалось
            # вдруг. Для самолётных рейсов эти данные не нужны.
            'supplier_code': None,
            'electronic_ticket': False,

            # Blablacar
            'number_of_variants': 1,
            'duration_in_seconds': 0,

            # Bus, Train
            'first_station': flight['station_from'],
            'last_station': flight['station_to'],
        }
    if flight['station_from']:
        ref['stations'].add(flight['station_from'])

    if flight['station_to']:
        ref['stations'].add(flight['station_to'])

    return key
