# -*- encoding: utf-8 -*-
from datetime import datetime

import json
import pytz
from marshmallow import fields
from marshmallow.base import FieldABC


class DatetimeAware(fields.Field):
    # todo: это нигде кроме тестов не используется.
    _LOCAL_DT_FMT = '%Y-%m-%d %H:%M:%S'

    def _serialize(self, value, attr, obj):
        if not value:
            return None

        return {
            'local': value.strftime(self._LOCAL_DT_FMT),
            'tzname': value.tzinfo.zone,
        }

    def _deserialize(self, value, attr, data):
        try:
            dt = datetime.strptime(value['local'], self._LOCAL_DT_FMT)
            tzinfo = pytz.timezone(value['tzname'])
            return tzinfo.localize(dt)
        except (ValueError, KeyError, pytz.UnknownTimeZoneError):
            self.fail('invalid')


class JsonField(fields.Field):
    """A json field, composed with another `Field` class or instance."""
    default_error_messages = {
        'invalid': 'Not a valid json.',
    }

    def __init__(self, cls_or_instance, serializer=None, **kwargs):
        self._json_serializer = serializer or json
        super(JsonField, self).__init__(**kwargs)
        if isinstance(cls_or_instance, type):
            if not issubclass(cls_or_instance, FieldABC):
                raise ValueError('The type of the json elements '
                                 'must be a subclass of '
                                 'marshmallow.base.FieldABC')
            self.container = cls_or_instance()
        else:
            if not isinstance(cls_or_instance, FieldABC):
                raise ValueError('The instances of the json '
                                 'elements must be of type '
                                 'marshmallow.base.FieldABC')
            self.container = cls_or_instance

    def _add_to_schema(self, field_name, schema):
        super(JsonField, self)._add_to_schema(field_name, schema)
        self.container.parent = self
        self.container.name = field_name

    def _serialize(self, value, attr, obj):
        dump = self.container._serialize(value, attr, obj)
        return self._json_serializer.dumps(dump)

    def _deserialize(self, value, attr, data):
        try:
            datum = json.loads(value)
        except (ValueError, TypeError):
            self.fail('invalid')
        return self.container.deserialize(datum)
