# -*- coding: utf-8 -*-
import json
import logging
from datetime import datetime
from typing import Any

import ujson
from decorator import decorator
from django.conf import settings
from flask import Blueprint, current_app, request
from marshmallow import Schema, fields
from werkzeug.exceptions import BadRequest, HTTPException

from travel.avia.library.python.ticket_daemon.query_validation.validation import validate_query, ValidationError

from travel.avia.ticket_daemon_api.environment import YANDEX_ENVIRONMENT_TYPE
from travel.avia.ticket_daemon_api.jsonrpc.controllers.check_sentry import (sentry_check_log_error, sentry_check_log_traceback)
from travel.avia.ticket_daemon_api.jsonrpc.handlers.scheme import InitQuerySchema
from travel.avia.ticket_daemon_api.jsonrpc.jsend import JsendError, JsendFail, JsendSuccess
from travel.avia.ticket_daemon_api.jsonrpc.lib.fare_families.fare_families import get_data_for_all_companies
from travel.avia.ticket_daemon_api.jsonrpc.lib.internal_daemon_client import InternalDaemonException
from travel.avia.ticket_daemon_api.jsonrpc.lib.result.collector.test_context import parse_test_context_proto
from travel.avia.ticket_daemon_api.jsonrpc.lib.utils import str_to_bool
from travel.avia.ticket_daemon_api.jsonrpc.query import Query, init_search, QueryIsNotValid

log = logging.getLogger(__name__)

api = Blueprint('api', __name__)
front_api = Blueprint('front_api', __name__)

DT_FMT = '%Y-%m-%dT%H:%M:%S'
D_FMT = '%Y-%m-%d'
D_FMT_LEN = len(datetime.now().strftime(D_FMT))


@decorator
def jsend_view(fn, *args, **kwargs):
    try:
        try:
            data = fn(*args, **kwargs)
            status_code = 200
            if isinstance(data, tuple):
                data, status_code = data
            jsend_data = JsendSuccess(data, status_code)
        except InternalDaemonException as e:
            log.warning(e, exc_info=True)
            try:
                error = ujson.loads(e.response)
            except (ValueError, TypeError):
                error = e.response
            data = {
                'description': e.description,
                'error': error
            }
            status_code = 500 if (e.code == 403 or 500 <= e.code <= 599) else e.code
            jsend_data = JsendFail(data, status_code)
        except BadRequest as e:
            log.warning(e, exc_info=True)
            jsend_data = JsendFail(e.description, e.code)
        except HTTPException as e:
            log.warning(e, exc_info=True)
            jsend_data = JsendFail(e.description, e.code)
        return jsend_response(jsend_data)
    except Exception as e:
        try:
            log.exception(e)
            return jsend_response(JsendError(str(e)))
        except Exception as e:
            log.critical('Error on send jsend error: %r', e)
            return jsend_response(JsendError('Server error'))


def jsend_response(jsend_data):
    return current_app.response_class(
        ujson.dumps(
            jsend_data.to_dict(),
            indent=4 if settings.DEBUG else 0,
        ),
        status=jsend_data.status_code,
        mimetype='application/json',
        content_type='application/json; charset=utf-8',
    )
    # security headers
    # response.headers.add('X-Frame-Options', 'SAMEORIGIN')
    # response.headers.add('X-XSS-Protection', '1; mode=block')
    # response.headers.add('X-Content-Type-Options', 'nosniff')


api.route('/system/sentry/check_log_error')(jsend_view(sentry_check_log_error))
api.route('/system/sentry/check_log_traceback')(jsend_view(sentry_check_log_traceback))


@api.route('/fare-families')
@jsend_view
def fare_families():
    return get_data_for_all_companies()


def clean_query_from_qid(qid):
    try:
        q = Query.from_qid(qid)
    except (ValueError, QueryIsNotValid) as e:
        log.warning('clean_query_from_qid: %r %r', qid, e)
        raise BadRequest(repr(e))
    try:
        validate_query(q)
    except ValidationError as e:
        raise BadRequest(e.data)
    return q


class YaUserSchema(Schema):
    uid = fields.Str()
    login = fields.Str()


class DjangoUserSchema(Schema):
    username = fields.Str()
    is_staff = fields.Bool()


class UserInfoSchema(Schema):
    yandexuid = fields.Str()
    passportuid = fields.Str()
    userip = fields.Str()

    yauser = fields.Nested(YaUserSchema)
    django_user = fields.Nested(DjangoUserSchema)


def _init_search_by_query(q):
    try:
        user_info = UserInfoSchema().load(json.loads(request.values['user_info']))[0]
    except:
        user_info = {}

    # https://st.yandex-team.ru/RASPTICKETS-21483
    if q.test_context and YANDEX_ENVIRONMENT_TYPE != 'production':
        tk = parse_test_context_proto(q.test_context)
        if tk.MockAviaVariants:
            return {
                'id': q.id,
                'queries': [],
            }

    data = init_search(
        q,
        ignore_cache=str_to_bool(request.values.get('ignore_cache')),
        user_info=user_info,
        test_id=request.values.get('testId')
    )

    return {
        'id': data['qid'],
        'queries': data.get('queries', []),
        'isDirectFlightsAvailable': data.get('isDirectFlightsAvailable', False),
    }


def make_query(data):
    # type: (Any) -> Query
    query_params, errors = InitQuerySchema().load(data)
    if errors:
        log.warning('Get QID. Query params are not valid: %r', errors)
        raise BadRequest(repr(errors))

    query = Query(**query_params)
    try:
        validate_query(query)
    except ValidationError as e:
        raise BadRequest(e.data)
    return query


@front_api.route('/check_search_query/')
@jsend_view
def front_check_search_query():
    make_query(request.values)


@front_api.route('/init_search_by_query/')
@jsend_view
def front_search_init_by_query():
    q = make_query(request.values)
    return _init_search_by_query(q)


# deprecated, pls use front/init_search_by_query
@front_api.route('/init_search/')
@jsend_view
def front_search_init():
    q = clean_query_from_qid(request.values.get('qid'))
    return _init_search_by_query(q)


def clean_query_from_key(qkey, service, lang):
    try:
        query = Query.from_key(qkey, service, lang)
    except Exception as e:
        log.warning(u'Bad query key %s: %r', repr(qkey), e)
        raise
    try:
        validate_query(query)
    except ValidationError as e:
        raise BadRequest(e.data)
    return query
