# -*- coding: utf-8 -*-
import json
import logging
import os

import ydb

from travel.library.python.tvm_ticket_provider import DeployTvmClient, TvmTicketProvider
from travel.avia.ticket_daemon_api.environment import YANDEX_ENVIRONMENT_TYPE, env_variable_provider


YDB_DATABASE = env_variable_provider.get('YDB_DATABASE')
YDB_ENDPOINT = env_variable_provider.get('YDB_ENDPOINT')

WIZARD_YDB_DATABASE = env_variable_provider.get('WIZARD_YDB_DATABASE')
WIZARD_YDB_ENDPOINT = env_variable_provider.get('WIZARD_YDB_ENDPOINT')


class ClientTvmConfig(object):
    """
    Parsed TVM config
    DEPLOY_TVM_CONFIG = {
      "BbEnvType": 0,
      "clients": {
        "client_name": {
          "secret": "SECRET",
          "self_tvm_id": 1,
          "dsts": {
            "dst_1_name": {"dst_id": 2},
            "dst_2_name": {"dst_id": 3}
          }
        }
      }
    }
    """

    def __init__(self):
        tvm_config = json.loads(os.getenv('DEPLOY_TVM_CONFIG'))
        assert len(tvm_config['clients']) == 1, 'Multiple clients is not allowed'
        client_name, client_info = tvm_config['clients'].items()[0]
        self.source_name = client_name
        self.self_tvm_id = client_info['self_tvm_id']
        self.destinations = {
            dst_name: dst_info['dst_id']
            for dst_name, dst_info in client_info['dsts'].items()
        }


def create_tvm_provider():
    logger = logging.getLogger('tvm_provider')
    tvm_token = os.getenv('TVMTOOL_LOCAL_AUTHTOKEN')
    tvm_config = ClientTvmConfig()
    tvm_client = DeployTvmClient(
        source_id=tvm_config.source_name,
        token=tvm_token,
        logger=logger,
        timeout=float(env_variable_provider.get('TVM_TIMEOUT', required=False, default=0.01)),
    )

    return TvmTicketProvider(
        client=tvm_client,
        destinations=tvm_config.destinations.values(),
        logger=logger,
    )


def auth_credentials():
    YDB_TOKEN = env_variable_provider.get('YDB_TOKEN', required=False)
    return ydb.AuthTokenCredentials(YDB_TOKEN)


def tvm_credentials():
    class YdbTvmCredentials(ydb.credentials.Credentials):

        def __init__(self, tvm_provider):
            self._tvm_provider = tvm_provider

        def auth_metadata(self):
            return [
                (ydb.credentials.YDB_AUTH_TICKET_HEADER, self._tvm_provider.get_ticket('ydb'))
            ]

    return YdbTvmCredentials(create_tvm_provider())


def driver_config(endpoint, database):
    if YANDEX_ENVIRONMENT_TYPE in {'testing', 'production'}:
        credentials_ = tvm_credentials()
    else:
        credentials_ = auth_credentials()

    return ydb.DriverConfig(
        endpoint,
        database=database,
        credentials=credentials_,
    )


DRIVER_CONFIG = driver_config(YDB_ENDPOINT, YDB_DATABASE)
WIZARD_DRIVER_CONFIG = driver_config(WIZARD_YDB_ENDPOINT, WIZARD_YDB_DATABASE)
SESSION_POOL_SIZE = 100
REMOVE_PARTNERS_WITH_EMPTY_RESPONSES_TIMEOUT = 5


def RECREATE_YDB_SESSION_POOL():
    from travel.avia.ticket_daemon_api.jsonrpc.lib import feature_flags
    return feature_flags.recreate_ydb_session_pool()
