# -*- coding: utf-8 -*-

from collections import namedtuple
from itertools import chain

from travel.avia.ticket_daemon_processing.pretty_fares.internal_logic.point_key import PointKey


class DirectionKey(
    namedtuple(
        'DirectionKey',
        ['from_point_key', 'to_point_key', 'tariff', 'adults', 'children', 'infants', 'national_version']
    )
):
    def __repr__(self):
        return self.to_string()

    @classmethod
    def from_raw_parts(cls, from_point_key, to_point_key, tariff, adults, children, infants, national_version):
        # type: (str, str, str, str, str, str, str) -> DirectionKey

        return DirectionKey(
            PointKey.parse(from_point_key),
            PointKey.parse(to_point_key),
            tariff,
            int(adults),
            int(children),
            int(infants),
            national_version
        )

    @classmethod
    def parse(cls, raw_direction_key):
        # type: (str) -> DirectionKey

        return DirectionKey.from_raw_parts(*raw_direction_key.split('_'))

    @classmethod
    def from_qkey(cls, qkey):
        # type: (str) -> DirectionKey

        direction_key_parts = qkey.split('_')
        return DirectionKey.from_raw_parts(*chain(direction_key_parts[:2], direction_key_parts[4:]))

    @classmethod
    def from_qid(cls, qid):
        # type: (str) -> DirectionKey

        return DirectionKey.from_qkey(qid.split('.')[-2])

    def to_string(self):
        # type: () -> str

        return (
            '{from_settlement}_{to_settlement}_{tariff}_'
            '{adults}_{children}_{infants}_{national_version}'.format(
                from_settlement=self.from_point_key.to_string(),
                to_settlement=self.to_point_key.to_string(),
                tariff=self.tariff,
                adults=str(self.adults),
                children=str(self.children),
                infants=str(self.infants),
                national_version=self.national_version,
            )
        )
