# -*- coding: utf-8 -*-

from enum import Enum
from typing import Optional


class PointKeyType(Enum):
    Station = 1
    Settlement = 2
    Region = 3
    Country = 4

    Other = 100


class PointKeyTypeParser(object):
    _string_to_point_key_type = {
        's': PointKeyType.Station,
        'c': PointKeyType.Settlement,
        'r': PointKeyType.Region,
        'l': PointKeyType.Country,
    }

    @classmethod
    def parse(cls, raw):
        # type: (str) -> PointKeyType

        return cls._string_to_point_key_type.get(raw, PointKeyType.Other)


class PointKeyTypeFormatter(object):
    _type_to_string_mapping = {
        PointKeyType.Station: 's',
        PointKeyType.Settlement: 'c',
        PointKeyType.Region: 'r',
        PointKeyType.Country: 'l',
    }

    @classmethod
    def to_string(cls, point_key_type):
        # type: (PointKeyType) -> str

        return cls._type_to_string_mapping.get(point_key_type, '?')


class PointKey(object):
    def __init__(self, type, id):
        # type: (PointKeyType, int) -> None

        self.type = type
        self.id = id

    @classmethod
    def parse(cls, raw):
        # type: (str) -> Optional[PointKey]

        if len(raw) < 2:
            return None

        return PointKey(PointKeyTypeParser.parse(raw[0]), int(raw[1:]))

    def of_type(self, point_key_type):
        # type: (PointKeyType) -> bool

        return self.type == point_key_type

    def is_station(self):
        # type: () -> bool

        return self.of_type(PointKeyType.Station)

    def is_settlement(self):
        # type: () -> bool

        return self.of_type(PointKeyType.Settlement)

    def to_string(self):
        # typ: () -> str

        return PointKeyTypeFormatter.to_string(self.type) + str(self.id)
