# coding=utf-8
import logging
import time
from contextlib import closing, contextmanager
from itertools import count

import six
from django.conf import settings
from ylock.backends.zookeeper import BackendConnectionError, Manager

from travel.avia.library.python.common.lib.timer import Timeline, TskvTimeline

log = logging.getLogger(__name__)

# disable ylock.backend.zookeeper logger (sentry anti-spam)
ylog_logger = logging.getLogger('ylock.backends.zookeeper')
ylog_logger.propagate = False
ylog_logger.disabled = True
ylog_logger.handlers = []


@contextmanager
def lock(key, prefix='/pretty_fares', ttl=60, acquire_timeout=30, timeline=None):
    # type: (str, str, int, int, Timeline) -> None

    timeline = (timeline or TskvTimeline(log)).with_defaults(key=key)

    with closing(
        Manager(
            hosts=settings.ZOOKEEPER_HOSTS,
            prefix=prefix,
            connect_timeout=2,
        )
    ) as manager:
        timeline.event('Waiting for the lock')

        timestamp = time.time() + acquire_timeout

        try:
            for retry in count(1):
                try:
                    with manager.lock(key, timeout=ttl, block=False) as locked:
                        if not locked:
                            timeline.event("Couldn't acquire the lock", retry=retry)
                            if time.time() < timestamp:
                                time.sleep(0.5)
                                continue
                            raise Exception("Couldn't acquire the lock for {!s}".format(key))

                        timeline.event('Acquired the lock', retry=retry)
                        try:
                            yield
                        finally:
                            timeline.event('Releasing the lock')
                except BackendConnectionError as e:
                    if time.time() < timestamp:
                        time.sleep(0.5)
                        continue
                    log.exception('Couldn\'t acquire the lock. %s. %s', retry, type(e))
                    six.raise_from(
                        Exception('Couldn\'t acquire the lock for {!s}'.format(key)),
                        e,
                    )
                break
        finally:
            timeline.event('Released the lock')
