# -*- coding: utf-8 -*-
from functools import partial
from itertools import ifilter
from operator import itemgetter

from travel.avia.ticket_daemon_processing.pretty_fares.models_utils.popularities import popularity


def n_filter(filters, iterable):
    return ifilter(lambda t: all(f(t) for f in filters), iterable)


def filter_interesting_variants(variants):
    """
    1) Выбираем группы предложений:
        - минимальная цена
        - популярный
        - прямой
        - с признаком "Авиакомпания"
        - (количество пересадок: n+1, если есть прямые)
        - (количество пересадок: n+2, если нет прямых)
    """

    variants = sorted(variants, key=itemgetter('national_tariff_price'))

    if not variants:
        return []

    def filtering_segments_count(segment_count):
        """
        Количество сегментов для фильтра равно n+1, если есть прямые
        варианты, и n+2 без прямых, где n - минимальное число сегментов.
        :return: minimum count of segments for filtering
        """
        return 2 if segment_count == 1 else segment_count + 2

    def min_segment_count(direction):
        return min(len(v[direction]) for v in variants)

    min_fwd_segments = min_segment_count('forward_segments')
    min_bwd_segments = min_segment_count('backward_segments')

    forward_segment_count = filtering_segments_count(min_fwd_segments)
    backward_segment_count = filtering_segments_count(min_bwd_segments)

    def filter_best_price(variant):
        return variant['national_tariff_price'] == variants[0]['national_tariff_price']

    def filter_by_segment_count(variant, fwd_count, bwd_count):
        return (
            len(variant['forward_segments']) <= fwd_count and
            len(variant['backward_segments']) <= bwd_count
        )

    def filter_by_aviacompany(variant):
        return variant['is_aviacompany']

    def filter_by_top_flights(variant):
        return popularity(variant['forward_segments']) + popularity(variant['backward_segments']) > 2

    filter_direct_flights = partial(
        filter_by_segment_count, fwd_count=1, bwd_count=1
    )
    filter_allowable_segments_count = partial(
        filter_by_segment_count, fwd_count=forward_segment_count, bwd_count=backward_segment_count
    )

    filters = (
        filter_best_price,
        filter_by_top_flights,
        filter_direct_flights,
        filter_by_aviacompany,
        filter_allowable_segments_count
    )

    return [
        variant
        for variant in variants
        if any(f(variant) for f in filters)
    ]
