from logging import getLogger

from cachetools.func import ttl_cache
import requests
from django.conf import settings
from retrying import retry

from travel.avia.library.python.avia_data.models import Currency

log = getLogger(__name__)


class TicketDaemonException(Exception):
    def __init__(self, message):
        super(TicketDaemonException, self).__init__(message)


@retry(stop_max_attempt_number=3, wait_fixed=10 * 1000)  # 3 attempts with 10 seconds delay
def get_full_results(qid):
    r = requests.get(
        url='http://{host}/jsendapi/td_processing/v1/results/{qid}'.format(
            host=settings.TICKET_DAEMON_HOST,
            qid=qid,
        ),
        timeout=settings.JSENDAPI_TIMEOUT
    )

    if r.status_code != 200:
        raise TicketDaemonException("Unsuccessful response from ticket_daemon_api. Response code: %r" % r.status_code)

    return r.json(), r.text


@ttl_cache(maxsize=None, ttl=60 * 60)
def _codes_by_id():
    return dict(Currency.objects.values_list('id', 'code'))


def currency_code_by_id(currency_id):
    return _codes_by_id().get(currency_id)


@ttl_cache(maxsize=None, ttl=60 * 60)
def backend_currency_rates(national_version):
    return requests.get(
        'http://{host}/rest/currencies/rates/{national_version}/ru'.format(
            host=settings.BACKEND_HOST,
            national_version=national_version,
        )
    ).json()['data']


@ttl_cache(maxsize=None, ttl=10 * 60)
def tariff_converter(national_version):
    backend_data = backend_currency_rates(national_version)

    base_currency = currency_code_by_id([backend_data['base_currency_id']])
    rates = {
        currency_code_by_id(rate['currency_id']): rate['rate']
        for rate in backend_data['rates']
    }

    def national(tariff):
        try:
            return {
                'currency': base_currency,
                'value': tariff['value'] * rates[tariff['currency']],
            }
        except Exception:
            log.exception("Can't convert [%s] %r", national_version, tariff)
            return None

    return national
