# coding=utf-8
import calendar
import logging
from datetime import datetime, timedelta
from functools import wraps
from time import time

import celery
import pytz
import raven

from billiard.process import current_process
from django.conf import settings
from raven.contrib.celery import register_signal, register_logger_signal

from travel.avia.library.python.common.lib.timer import TskvTimeline
from travel.avia.ticket_daemon_processing.pretty_fares.errors_handling.safe_run import safe_run
from travel.avia.ticket_daemon_processing.pretty_fares.internal_logic.direction_key import DirectionKey
from travel.avia.ticket_daemon_processing.pretty_fares.messaging.daemon_filter import filter_interesting_variants
from travel.avia.ticket_daemon_processing.pretty_fares.messaging.daemon_tasks import (
    get_full_results, backend_currency_rates, currency_code_by_id
)
from travel.avia.ticket_daemon_processing.pretty_fares.messaging.db_tasks import send_to_db
from travel.avia.ticket_daemon_processing.pretty_fares.messaging.wizard_tasks import send_to_wizard_cache, disabled_pcodes, national_version_from_qkey
from travel.avia.ticket_daemon_processing.pretty_fares.models_utils.popularities import popularity
from travel.avia.ticket_daemon_processing.pretty_fares.internal_logic.direction_cheapest_updater import direction_cheapest_updater
from travel.avia.ticket_daemon_processing.pretty_fares.messaging.daemon_tasks import log as daemon_log
from travel.avia.ticket_daemon_processing.pretty_fares.lib.feature_flags import save_minprices_to_airports
from travel.avia.ticket_daemon_processing.pretty_fares.settings import VARIANT_TTL_IN_HOURS


class Celery(celery.Celery):

    def on_configure(self):
        if settings.SENTRY_DSN:
            client = raven.Client(settings.SENTRY_DSN)
            register_logger_signal(client)
            register_signal(client)


celery_app = Celery('variants_saver')
celery_app.config_from_object('travel.avia.ticket_daemon_processing.pretty_fares:settings')

logger = logging.getLogger(__name__)
traceback_log = logging.getLogger('main.traceback')


def logged_exceptions(func):
    @wraps(func)
    def wrapper(*args, **kwargs):
        try:
            return func(*args, **kwargs)
        except Exception as exc:
            traceback_log.exception('%r', exc)
            raise

    return wrapper


@celery_app.task()
@logged_exceptions
def store_processed_qid(qid, timestamp, meta=None):
    try:
        full_results, json_result = get_full_results(qid)
    except Exception as e:
        logger.exception("Got exception: %r.", e)
        return

    store_variants_to_price_index.apply_async(kwargs={
        'qid': qid,
        'json_result': json_result,
    })


# Делаем, чтобы таск был под старым именем.
# Первая строчка для того, чтобы при создании таска в очередь записать старое имя.
# Вторая строчка для того, чтобы при чтении таска из очереди celery нашел его.
store_processed_qid.name = 'pretty_fares.messaging.tasks.store_processed_qid'
celery_app.tasks['pretty_fares.messaging.tasks.store_processed_qid'] = celery_app.tasks[
    'travel.avia.ticket_daemon_processing.pretty_fares.messaging.tasks.store_processed_qid']


@celery_app.task(bind=True, max_retries=50)
@logged_exceptions
def store_wizard_variants(self, qid, timestamp, results):
    try:
        _store_wizard_variants(qid, timestamp, results)
    except Exception as exc:
        self.retry(
            kwargs={
                'qid': qid,
                'timestamp': timestamp,
                'results': results,
            },
            countdown=2,
            exc=exc
        )
        logger.exception(
            'Exception %r on store_wizard_variants. Retrying %s', exc, qid
        )


@safe_run(timeline=TskvTimeline(daemon_log))
def _store_wizard_variants(qid, timestamp, results):
    task_id = celery_app.current_worker_task.request.id
    worker = current_process().name

    processing_start = time()
    utcnow = datetime.now(pytz.UTC)

    timeline = TskvTimeline(daemon_log)

    timeline.event('Task started', id=task_id, worker=worker)

    qkey = qid.split('.')[-2]
    direction_key = DirectionKey.from_qkey(qkey)

    iatas = {s['id']: s['code'] for s in results['data']['reference']['stations']}

    flights = {
        f['key']: {
            'number': f['number'],
            'acRaspId': f.get('aviaCompany') or f['company'],
            'acIata': f['number'].split()[0],
            'flightNumber': f['number'].split()[1],
            'arrDt': f['arrival']['local'],
            'depDt': f['departure']['local'],
            'arrTzName': f['arrival']['tzname'],
            'depTzName': f['departure']['tzname'],
            'depRaspId': f['from'],
            'arrRaspId': f['to'],
            'depIata': iatas.get(f['from']),
            'arrIata': iatas.get(f['to']),
        }
        for f in results['data']['reference']['flights'] if f['departure'] and f['arrival']
    }

    offer_time = pytz.UTC.localize(datetime.strptime(timestamp, '%Y-%m-%d %H:%M:%S'))

    variants = []
    for fare in results['data']['variants']['fares']:
        try:
            forward = [flights[key] for key in fare['route'][0]]
            backward = [flights[key] for key in fare['route'][1]]
        except KeyError:
            continue

        variant_popularity = popularity(forward) + popularity(backward)

        for price in fare['prices']:
            variants.append({
                'forward_segments': forward,
                'backward_segments': backward,
                'tariff_currency': price['tariff']['currency'],
                'tariff_price': price['tariff']['value'],
                'partner': price['partnerCode'],
                'queryTime': price['queryTime'],
                'is_aviacompany': price.get('fromCompany', False),
                'popularity': variant_popularity,
                'baggage': price['baggage'],
                'expires_at': offer_time + timedelta(hours=VARIANT_TTL_IN_HOURS),
                'created_at': offer_time,
            })

    timeline.event('Prepared variants', count=len(variants))

    # В колдунщик должны попадать только варианты с валютой для запрошенной нацверсии.
    # Переводить валюты нельзя. RASPTICKETS-12601
    national_currency = currency_code_by_id(
        backend_currency_rates(national_version=qkey.split('_')[-1])['base_currency_id']
    )
    variants = [v for v in variants if v['tariff_currency'] == national_currency]
    for v in variants:
        v['national_tariff_currency'] = v['tariff_currency']
        v['national_tariff_price'] = v['tariff_price']

    timeline.event('Left only national currency', currency=national_currency, count=len(variants))

    nv = national_version_from_qkey(qkey)
    variants = [v for v in variants if v['partner'] not in disabled_pcodes(nv)]
    timeline.event('Left only allowed partners', national_version=nv, count=len(variants))

    interesting_variants = filter_interesting_variants(variants)

    timeline.event('Left only interesting variants', count=len(interesting_variants))

    send_to_wizard_cache(
        variants=interesting_variants,
        query={
            'qid': qid,
            'offer_time': calendar.timegm(offer_time.utctimetuple()),
            'qkey': qkey,
        },
        timeline=timeline,
        utcnow=utcnow,
        processing_start=processing_start
    )

    necessary_to_save_minprices_to_airports = save_minprices_to_airports()
    timeline.event('Save minprice to airports: ' + str(necessary_to_save_minprices_to_airports))
    if necessary_to_save_minprices_to_airports:
        direction_cheapest_updater.update_direction_cheapest_with_airports(direction_key, variants, qkey, utcnow, timeline=timeline)

    timeline.event('Task finished', id=task_id, worker=worker)


@celery_app.task()
@logged_exceptions
def store_variants_to_price_index(qid, json_result):
    qkey = qid.split('.')[-2]

    send_to_db(
        query=qkey,
        json_result=json_result,
    )


store_variants_to_price_index.name = 'pretty_fares.messaging.tasks.store_variants_to_price_index'

# Делаем, чтобы таск был под старым именем.
# Первая строчка для того, чтобы при создании таска в очередь записать старое имя.
# Вторая строчка для того, чтобы при чтении таска из очереди celery нашел его.
store_variants_to_price_index.name = 'pretty_fares.messaging.tasks.store_variants_to_price_index'
celery_app.tasks['pretty_fares.messaging.tasks.store_variants_to_price_index'] = celery_app.tasks[
    'travel.avia.ticket_daemon_processing.pretty_fares.messaging.tasks.store_variants_to_price_index']
