# -*- coding: utf-8 -*-
import json
import logging
import random
from time import time
from datetime import timedelta
from itertools import islice
from operator import itemgetter
from typing import List, Dict, Any

from cachetools.func import ttl_cache
import dateutil.parser
from django.conf import settings

from travel.avia.library.python.common.models.partner import Partner, DohopVendor
from travel.avia.library.python.common.utils.iterrecipes import pairwise, group_by
from travel.avia.ticket_daemon_processing.pretty_fares.internal_logic.direction_cheapest_updater import update_direction_cheapest_old
from travel.avia.ticket_daemon_processing.pretty_fares.internal_logic.direction_key import DirectionKey
from travel.avia.ticket_daemon_processing.pretty_fares.models_utils.geo import (
    get_settlement_id_by_code, get_station_id_by_code
)
from travel.avia.ticket_daemon_processing.pretty_fares.saas import retrying_json_saas as json_saas
from travel.avia.ticket_daemon_processing.pretty_fares.settings import VARIANT_TTL_IN_HOURS


log = logging.getLogger(__name__)
metrics_logger = logging.getLogger('wizard_tasks_metrics')


class VariantTypes(object):
    Cheap = 'min_price'
    AviaCompany = 'aviacompany'
    Popular = 'top_flight'


PESSIMIZED_PARTNERS = {'ozon', 'sindbad'}

MIN_PRICE_MAX = 1
POPULARS_MAX = 19

TIMEOUT = 15.0


def send_to_wizard_cache(variants, query, timeline, utcnow, processing_start):
    """
    Запись вариантов в SaaS, чтобы потом показывать их в колдунщике.
    Варианты не всегда приходят выдачей ото всех партнёров, так как протухают
    неравномерно, и демон может инициировать поиски только для части партнёров.
    Нужно обеспечить их правильное сливание.

    В эту функцию попадает выдача после завершения опроса queried_partners.
    """
    qkey = query['qkey']
    direction_key = DirectionKey.from_qkey(qkey)
    timeline.event('Direction key', key=direction_key.to_string())

    timeline = timeline.with_defaults(qkey=qkey)

    timeline.event('New send_to_wizard_cache task', **query)

    cache_key = '{}/{}'.format(settings.WIZARD_CACHE_PREFIX, qkey)

    timeline.event('Current partners', codes=sorted(set(v['partner'] for v in variants)))

    if not variants:
        timeline.event('No variants provided')
        json_saas.delete(cache_key)
        update_direction_cheapest_old(direction_key, [], qkey, utcnow, timeline)
        timeline.event('Deleted variants from SaaS', key=cache_key)
        return

    variant_counts = {
        pcode: len(group) for pcode, group in group_by(variants, itemgetter('partner'))
    }
    timeline.event('Variant counts', **variant_counts)

    for v in variants:
        v.update({
            'airport_changes': airport_changes(v),
            'qid': query['qid'],
        })

    timeline.event('Filled raw variants', count=len(variants))

    timeline.event(
        'Prices info',
        min=min(v['national_tariff_price'] for v in variants),
        max=max(v['national_tariff_price'] for v in variants),
    )

    variants = [v for v in variants if v['expires_at'] > utcnow]
    timeline.event('Left only actual variants', utcnow=utcnow, count=len(variants))

    variants = deduplicated(variants)
    timeline.event('Deduplicated variants', count=len(variants))

    variants = typed_wizard_variants(
        variants,
        cheaps_max=MIN_PRICE_MAX,
        populars_max=POPULARS_MAX,
    )
    timeline.event(
        'Chose variants for wizard',
        count=len(variants),
        winners=','.join(v['partner'] for v in variants),
    )

    expiration_dt = (
        max(v['expires_at'] for v in variants)
        if variants else
        utcnow + timedelta(hours=VARIANT_TTL_IN_HOURS)
    )

    json_saas.index(
        key='{}/{}'.format(settings.WIZARD_CACHE_PREFIX, qkey),
        doc={
            'variants': variants,
            'uniques': max(variant_counts.itervalues()) if variants else 0,
            'expires_at': expiration_dt,
        },
        expires_at=expiration_dt,
        timeout=TIMEOUT,
    )

    update_direction_cheapest_old(direction_key, variants, qkey, utcnow, timeline)

    _write_processing_metrics(processing_start, processing_finish=time())

    timeline.event('Finish')


def _write_processing_metrics(processing_start, processing_finish):
    # type: (float, float) -> None

    duration_ms = (processing_finish - processing_start) * 1000
    metric_data = {
        'unixtime': int(processing_finish),
        'duration': int(duration_ms)
    }

    metrics_logger.info(json.dumps(metric_data))


def deduplicated(variants):
    """
    Удаляем дубликаты вариантов.
    Среди одинаковых перелётов выбираем самый дешёвый, но
    при совпадении популярностей выбираем вариант от АК.
    """
    def identity(variant):
        def segments_key(segments):
            return tuple(
                (s['depDt'], s['arrDt'], s['depRaspId'], s['arrRaspId']) for s in (segments or [])
            )
        return (
            segments_key(variant['forward_segments']),
            segments_key(variant['backward_segments']),
        )

    def best(group):
        min_price = min(v['national_tariff_price'] for v in group)
        group = [v for v in group if v['national_tariff_price'] == min_price]

        if any(v['is_aviacompany'] for v in group):
            group = [v for v in group if v['is_aviacompany']]

        # По возможности выбираем не пессимизированных партнёров. Если не можем выбрать, рандом.
        candidates = {v['partner']: v for v in group}
        winner = random.choice(
            [p for p in candidates if p not in PESSIMIZED_PARTNERS] or candidates.keys()
        )

        return candidates[winner]

    return [
        best(group)
        for _, group in group_by(variants, key=identity)
    ]


def deserialized(saas_variants):
    """
    После десериализации из SaaS варианты имеют строковые поля вместо datetime.
    Исправляем только expires_at.
    """
    result = [v.copy() for v in saas_variants]
    for v in result:
        v['expires_at'] = dateutil.parser.parse(v['expires_at'])
    return result


def airport_changes(variant):
    """
    Дополнение варианта признаком смены аэропорта
    """
    def transfers(segments):
        for segment1, segment2 in pairwise(segments):
            if segment1['arrIata'] != segment2['depIata']:
                transfer = {
                    'airport_from': get_station_id_by_code(segment1['arrIata']),
                    'airport_to': get_station_id_by_code(segment2['depIata']),
                    'settlement': get_settlement_id_by_code(segment1['arrIata']),
                }
                if all(transfer.values()):  # ensure all fields are not None
                    yield transfer

    return (
        list(transfers(variant['forward_segments'])) +
        list(transfers(variant['backward_segments']))
    )


def typed_wizard_variants(variants, cheaps_max, populars_max):
    # type: (List[Dict[str, Any]], int, int) -> List[Dict[str, Any]]

    """
    Отбираем:
    1. Самые дешевые варианты;
    2. Самые популярные из оставшихся.
    При совпадении ключа сравнения предпочитаем самые дешёвые варианты.
    """
    variants = [v.copy() for v in variants]
    for v in variants:
        v.pop('type', None)

    cheapest = sorted(variants, key=itemgetter('national_tariff_price'))
    # Пользуемся стабильностью сортировки
    popular = sorted(cheapest, key=itemgetter('popularity'), reverse=True)

    result = []

    for v in cheapest[:cheaps_max]:
        v['type'] = VariantTypes.Cheap
        result.append(v)

    populars = (v for v in popular if 'type' not in v)
    for v in islice(populars, 0, populars_max):
        v['type'] = VariantTypes.Popular
        result.append(v)

    return result


def national_version_from_qkey(qkey):
    return qkey.split('_')[8]


@ttl_cache(maxsize=None, ttl=60)
def disabled_pcodes(national_version):
    partners = Partner.objects.all()
    vendors = DohopVendor.objects.all()
    dohop_meta = Partner.objects.get(code='dohop')

    def disabled(model):
        return not model.enabled or not model.enabled_in('wizard', national_version)

    models = filter(disabled, partners) + list(vendors if disabled(dohop_meta) else filter(disabled, vendors))
    return frozenset(p.code for p in models)
