# -*- coding: utf-8 -*-
from collections import defaultdict, OrderedDict
from itertools import chain
from typing import List, Dict, Optional

from cachetools.func import lru_cache

from travel.avia.library.python.common.models.geo import StationCode, Station, Station2Settlement, StationType
from travel.avia.library.python.common.models.transport import TransportType

_station_id_by_iata = dict(StationCode.objects.filter(
    system__code='iata').values_list('code', 'station_id'))
_station_id_by_sirena = dict(StationCode.objects.filter(
    system__code='sirena').values_list('code', 'station_id'))
_settlement_id_by_station_id = dict(Station.objects.filter(
    t_type_id=TransportType.PLANE_ID).values_list('id', 'settlement_id'))
_settlement_id_by_code = {
    code: _settlement_id_by_station_id.get(station_id) for code, station_id
    in chain(_station_id_by_iata.iteritems(), _station_id_by_sirena.iteritems())
}


class OrderedSet(OrderedDict):
    """
    В итоге чтобы получить значения, нужно будет преобразовать в список
    """
    def add(self, value):
        if value not in self:
            self[value] = None


def get_station_id_by_code(code):
    """

    :param code: iata or sirena
    :return:
    """
    station_id = (_station_id_by_iata.get(code) or
                  _station_id_by_sirena.get(code))
    if station_id:
        return station_id


def get_settlement_id_by_code(code):
    """

    :param code: iata or sirena
    :return:
    """
    return _settlement_id_by_code.get(code)


def airports_ids_by_settlement_id(settlement_id):
    # type: (int) -> List[int]

    return _airport_ids_by_settlement_id().get(settlement_id, [])


@lru_cache(maxsize=None)
def _station2settlement():
    # type: () -> Dict[long, long]

    return dict(
        Station2Settlement.objects.
            filter(station_id__in=_airport_to_settlement().keys()).
            values_list('station_id', 'settlement_id')
    )


@lru_cache(maxsize=None)
def _airport_to_settlement():
    # type: () -> Dict[int, Optional[int]]
    return dict(
        Station.objects.
            filter(station_type_id=StationType.AIRPORT_ID).
            values_list('id', 'settlement_id')
    )


@lru_cache(maxsize=None)
def _airport_ids_by_settlement_id():
    # type: () -> Dict[int, List[int]]

    by_settlement = defaultdict(OrderedSet)

    for airport_id, settlement_id in _airport_to_settlement().iteritems():
        if settlement_id:
            by_settlement[settlement_id].add(airport_id)

    for airport_id, settlement_id in _station2settlement().iteritems():
        by_settlement[settlement_id].add(airport_id)

    return {
        settlement_id: tuple(stations_ids)
        for settlement_id, stations_ids in by_settlement.iteritems()
    }


@lru_cache(maxsize=None)
def _settlements_ids_by_airport_id():
    # type: () -> Dict[int, List[int]]

    by_airport = defaultdict(OrderedSet)

    for airport_id, settlement_id in _airport_to_settlement().iteritems():
        if settlement_id:
            by_airport[airport_id].add(settlement_id)

    for airport_id, settlement_id in _station2settlement().iteritems():
        by_airport[airport_id].add(settlement_id)

    return {
        station_id: tuple(settlements_ids)
        for station_id, settlements_ids in by_airport.iteritems()
    }


def settlement_id_by_station_id(station_id):
    # type: (int) -> Optional[int]

    settlements_ids = _settlements_ids_by_airport_id().get(station_id, [])

    if settlements_ids:
        return settlements_ids[0]

    return None
