# -*- encoding: utf-8 -*-
import logging

from tornado.web import Application
from tornado_opentracing_helpers import setup_tracing

from travel.avia.travelers.application import settings
from travel.avia.travelers.application.handlers import PingHandler, VersionHandler, ShutdownHandler
from travel.avia.travelers.application.handlers.bonus_cards import BonusCardsHandler
from travel.avia.travelers.application.handlers.combine_passengers import CombinePassengersHandler
from travel.avia.travelers.application.handlers.document_types import DocumentTypesHandler
from travel.avia.travelers.application.handlers.documents import DocumentsHandler
from travel.avia.travelers.application.handlers.passengers import PassengersHandler
from travel.avia.travelers.application.handlers.traveler import TravelerHandler
from travel.avia.travelers.application.services.data_sync.client import create_datasync
from travel.avia.travelers.application.services.geodata.client import init_geo_data_client
from travel.avia.library.python.shared_objects import SharedFlag

logger = logging.getLogger()


class ToolsApplication(Application):
    pass


def create_application(shutdown_flag: SharedFlag, debug=False, data_sync_client=None, enable_tracing=True):
    data_sync_client = data_sync_client or create_datasync()
    init_geo_data_client()

    tornado_application_tracing_kwargs = {}
    if enable_tracing:
        tornado_application_tracing_kwargs = setup_tracing(
            service_name=settings.TRACING_SERVICE_NAME,
            version=settings.VERSION,
            include_env_variables=['DEPLOY_STAGE_ID', 'DEPLOY_NODE_DC', 'DEPLOY_BOX_ID'],
            traced_attributes_of_tornado_request=['method', 'path', 'query', 'query_arguments', 'body_arguments'],
            filter_paths=['/ping'],
            jaeger_sampler_type=settings.TRACING_JAEGER_SAMPLER_TYPE,
            jaeger_sampler_parameter=settings.TRACING_JAEGER_SAMPLER_PARAMETER,
        )

    handler_params = dict(data_sync_client=data_sync_client)
    availability_params = {'shutdown_flag': shutdown_flag}
    application = ToolsApplication(
        [
            (r'/ping', PingHandler, availability_params),
            (r'/shutdown', ShutdownHandler, availability_params),
            (r'/version', VersionHandler, handler_params),
            (r'/travelers/(\d+)', TravelerHandler, handler_params),
            (r'/travelers/(\d+)/passengers', PassengersHandler, handler_params),
            (
                r'/travelers/(\d+)/passengers/([0-9a-z]{8}\-[0-9a-z]{4}\-[0-9a-z]{4}\-[0-9a-z]{4}\-[0-9a-z]{12})',
                PassengersHandler,
                handler_params,
            ),
            (
                r'/travelers/(\d+)'
                r'/passengers/([0-9a-z]{8}\-[0-9a-z]{4}\-[0-9a-z]{4}\-[0-9a-z]{4}\-[0-9a-z]{12})'
                r'/documents',
                DocumentsHandler,
                handler_params,
            ),
            (
                r'/travelers/(\d+)'
                r'/passengers/([0-9a-z]{8}\-[0-9a-z]{4}\-[0-9a-z]{4}\-[0-9a-z]{4}\-[0-9a-z]{12})'
                r'/documents/([0-9a-z]{8}\-[0-9a-z]{4}\-[0-9a-z]{4}\-[0-9a-z]{4}\-[0-9a-z]{12})',
                DocumentsHandler,
                handler_params,
            ),
            (r'/document_types', DocumentTypesHandler, handler_params),
            (
                r'/travelers/(\d+)'
                r'/passengers/([0-9a-z]{8}\-[0-9a-z]{4}\-[0-9a-z]{4}\-[0-9a-z]{4}\-[0-9a-z]{12})'
                r'/bonus-cards',
                BonusCardsHandler,
                handler_params,
            ),
            (
                r'/travelers/(\d+)'
                r'/passengers/([0-9a-z]{8}\-[0-9a-z]{4}\-[0-9a-z]{4}\-[0-9a-z]{4}\-[0-9a-z]{12})'
                r'/bonus-cards/([0-9a-z]{8}\-[0-9a-z]{4}\-[0-9a-z]{4}\-[0-9a-z]{4}\-[0-9a-z]{12})',
                BonusCardsHandler,
                handler_params,
            ),
            (
                r'/travelers/(\d+)'
                r'/passengers/([0-9a-z]{8}\-[0-9a-z]{4}\-[0-9a-z]{4}\-[0-9a-z]{4}\-[0-9a-z]{12})'
                r'/combine',
                CombinePassengersHandler,
                handler_params,
            ),
        ],
        debug=debug,
        **tornado_application_tracing_kwargs
    )

    return application
