# -*- encoding: utf-8 -*-
import logging
import os
from typing import Optional, List, Union, Dict
from uuid import UUID

import tornado
from ticket_parser2.api.v1.exceptions import TicketParsingException
from tornado.web import RequestHandler, HTTPError
from travel.library.python.tvm_ticket_provider import provider_fabric

from travel.avia.travelers.application import settings
from travel.avia.travelers.application.models import Traveler, Passenger, Document, BonusCard
from travel.avia.travelers.application.yt_logging import yt_request_log
from travel.avia.library.python.shared_objects import SharedFlag

logger = logging.getLogger(__name__)
Documents = List[Union[Document, BonusCard]]


class BaseHandler(RequestHandler):
    def __init__(self, application, request, **kwargs):
        super(BaseHandler, self).__init__(application, request, **kwargs)
        self._check_user_ticket = False
        self._check_service_ticket = False
        self._user_ticket = None
        self._parsed_user_ticket = None
        self._request_service_id = None
        self._tvm_provider = provider_fabric.create(
            source_id=settings.TVM_SERVICE_ID,
            secret=settings.TVM_SECRET,
        )

    def initialize(self, data_sync_client):
        self._data_sync_client = data_sync_client

    @property
    def data_sync_client(self):
        return self._data_sync_client

    def prepare(self):
        self._check_tvm_user_ticket()
        self._check_tvm_service_ticket()
        super(BaseHandler, self).prepare()

    def _check_tvm_user_ticket(self):
        self._user_ticket = None
        if not self._check_user_ticket:
            return

        self._user_ticket = self.request.headers.get('X-Ya-User-Ticket')
        if not self._user_ticket:
            raise HTTPError(400, reason='Empty tvm user ticket')

        try:
            self._parsed_user_ticket = self._tvm_provider.check_user_ticket(self._user_ticket)
        except TicketParsingException as e:
            logger.error('TvmError: %s, %s', e.message, e.debug_info)
            raise HTTPError(401, reason='Tvm error: {}'.format(e.message))

    def _check_user_uid(self, user_uid):
        if not self._parsed_user_ticket or not self._parsed_user_ticket.check_uid(user_uid):
            raise HTTPError(403, reason='Wrong user uid')

    def _check_tvm_service_ticket(self):
        self._request_service_id = None
        if not self._check_service_ticket:
            return

        if 'X-Ya-Service-Ticket' not in self.request.headers:
            raise HTTPError(400, reason='Empty tvm service ticket')

        try:
            ticket = self._tvm_provider.check_service_ticket(self.request.headers.get('X-Ya-Service-Ticket'))
            self._request_service_id = ticket.src
        except TicketParsingException as e:
            logger.error('TvmError: %s, %s', e.message, e.debug_info)
            raise HTTPError(401, reason='Tvm error: {}'.format(e.message))

    def _get_fields(self):
        return self.get_argument('fields', '').split(',')

    def _get_traveler(self, uid: str, silent: bool = False) -> Optional[Traveler]:
        traveler = self.data_sync_client.get_traveler(uid, self._user_ticket)
        if not traveler:
            if silent:
                return None
            raise HTTPError(404, reason='Traveler not found')

        return traveler

    def _get_passenger(
        self, uid: str, passenger_id: str, fields: List[str] = None, silent=False
    ) -> Optional[Passenger]:
        passenger = self.data_sync_client.get_passenger(uid, self._user_ticket, passenger_id)
        if not passenger:
            if silent:
                return None
            raise HTTPError(404, reason='Passenger not found')

        passenger_fields = {}
        if fields:
            if 'documents' in fields:
                passenger_fields['documents'] = self.data_sync_client.get_passenger_documents(
                    uid, self._user_ticket, passenger
                )
            if 'bonus-cards' in fields:
                passenger_fields['bonus_cards'] = self.data_sync_client.get_passenger_bonus_cards(
                    uid, self._user_ticket, passenger
                )

        return passenger.fill(passenger_fields)

    @staticmethod
    def group_passengers_items_by_passenger_id(items: Documents, passengers: List[Passenger]) -> Dict[UUID, Documents]:
        result = {p.id: [] for p in passengers}
        for item in items:
            result[item.passenger_id].append(item)
        return result

    def _get_passengers(self, uid: str, fields: List[str] = None) -> List[Passenger]:
        passengers = self.data_sync_client.get_passengers(uid, self._user_ticket)
        passengers_fields = {}
        if fields:
            if 'documents' in fields:
                documents = self.data_sync_client.get_documents(uid, self._user_ticket)
                passengers_fields['documents'] = self.group_passengers_items_by_passenger_id(documents, passengers)
            if 'bonus-cards' in fields:
                bonus_cards = self.data_sync_client.get_bonus_cards(uid, self._user_ticket)
                passengers_fields['bonus_cards'] = self.group_passengers_items_by_passenger_id(bonus_cards, passengers)
            for p in passengers:
                p.fill({f: passengers_fields[f][p.id] for f in passengers_fields})
        return passengers

    def _get_document(
        self, uid: str, passenger: Passenger, document_id: str, silent: bool = False
    ) -> Optional[Document]:
        document = self.data_sync_client.get_document(uid, self._user_ticket, passenger, document_id)
        if not document:
            if silent:
                return None
            raise HTTPError(404, reason='Document not found')

        return document

    def _get_bonus_card(
        self, uid: str, passenger: Passenger, bonus_card_id: str, silent: bool = False
    ) -> Optional[BonusCard]:
        bonus_card = self.data_sync_client.get_bonus_card(uid, self._user_ticket, passenger, bonus_card_id)
        if not bonus_card:
            if silent:
                return None
            raise HTTPError(404, reason='Bonus card not found')

        return bonus_card

    def _yt_request_log(
        self,
        uid: str,
        traveler_id: str,
        passenger_id: Optional[str] = None,
        document_id: Optional[str] = None,
        bonus_card_id: Optional[str] = None,
    ):
        yt_request_log(
            service_id=self._request_service_id,
            method=self.request.method,
            uid=uid,
            traveler_id=traveler_id,
            passenger_id=passenger_id,
            document_id=document_id,
            bonus_card_id=bonus_card_id,
        )


class PingHandler(BaseHandler):
    def initialize(self, shutdown_flag: SharedFlag) -> None:
        self._shutdown_flag = shutdown_flag

    def get(self):
        if self._shutdown_flag.is_set():
            self.set_status(410)
            self.write('Shutdown')
        else:
            self.write('OK')


class ShutdownHandler(BaseHandler):
    def initialize(self, shutdown_flag: SharedFlag) -> None:
        self._shutdown_flag = shutdown_flag

    def post(self) -> None:
        self._shutdown_flag.set()
        self.set_header('Content-Type', 'text/plain')
        self.write('OK')


class VersionHandler(BaseHandler):
    def get(self):
        self.set_header('Content-Type', 'text/plain')
        self.write('Tornado version: {}\n'.format(tornado.version))
        self.write('Package tag: {}\n'.format(os.getenv('DEPLOY_DOCKER_IMAGE')))
        self.write('Package hash: {}\n'.format(os.getenv('DEPLOY_DOCKER_HASH')))
