# -*- encoding: utf-8 -*-
import logging
from typing import Optional

from marshmallow import ValidationError

from travel.avia.travelers.application.handlers import BaseHandler
from travel.avia.travelers.application.models import Passenger, BonusCard
from travel.avia.travelers.application.services.data_sync import data_sync_exceptions
from travel.avia.travelers.application.schemas import BonusCardSchema

logger = logging.getLogger(__name__)
bonus_card_schema = BonusCardSchema()
bonus_cards_schema = BonusCardSchema(many=True)


class BonusCardsHandler(BaseHandler):
    def __init__(self, application, request, **kwargs):
        super(BonusCardsHandler, self).__init__(application, request, **kwargs)
        self._check_user_ticket = True
        self._check_service_ticket = True

    @data_sync_exceptions(logger)
    def get(self, uid: str, passenger_id: str, bonus_card_id: Optional[str] = None):
        self._check_user_uid(uid)
        passenger = self._get_passenger(uid, passenger_id)

        if bonus_card_id is None:
            self.set_header('Content-Type', 'application/json')
            self.write(bonus_cards_schema.dumps(
                self.data_sync_client.get_passenger_bonus_cards(uid, self._user_ticket, passenger)
            ))
        else:
            self.write(bonus_card_schema.dump(
                self._get_bonus_card(uid, passenger, bonus_card_id)
            ))

    @data_sync_exceptions(logger)
    def post(self, uid: str, passenger_id: str, bonus_card_id: Optional[str] = None):
        self._check_user_uid(uid)
        if bonus_card_id:
            return self.send_error(405, reason='Use PUT for update bonus card')
        self._update_bonus_card(uid, self._get_passenger(uid, passenger_id), BonusCard())

    @data_sync_exceptions(logger)
    def put(self, uid: str, passenger_id: str, bonus_card_id: Optional[str] = None):
        self._check_user_uid(uid)
        if not bonus_card_id:
            return self.send_error(404, reason='Empty bonus card id')

        passenger = self._get_passenger(uid, passenger_id)
        self._update_bonus_card(
            uid,
            passenger,
            self._get_bonus_card(uid, passenger, bonus_card_id),
        )

    @data_sync_exceptions(logger)
    def delete(self, uid: str, passenger_id: str, bonus_card_id: Optional[str] = None):
        self._check_user_uid(uid)
        passenger = self._get_passenger(uid, passenger_id)
        bonus_card = self._get_bonus_card(uid, passenger, bonus_card_id)

        self.data_sync_client.delete_bonus_card(uid, self._user_ticket, bonus_card)

        self.set_status(201, 'Deleted')

        self._yt_request_log(
            uid=uid,
            traveler_id=passenger.traveler_id,
            passenger_id=passenger.id,
            bonus_card_id=bonus_card.id,
        )

    def _update_bonus_card(self, uid: str, passenger: Passenger, bonus_card: BonusCard):
        try:
            data = bonus_card_schema.loads(self.request.body.decode('utf8'))
            data['passenger_id'] = passenger.id

            bonus_card = self.data_sync_client.save_bonus_card(
                uid,
                self._user_ticket,
                passenger,
                bonus_card.fill(data),
            )

            self.write(bonus_card_schema.dump(bonus_card))

            self._yt_request_log(
                uid=uid,
                traveler_id=passenger.traveler_id,
                passenger_id=passenger.id,
                bonus_card_id=bonus_card.id,
            )

        except ValueError:
            return self.send_error(400, reason='Wrong request')
        except ValidationError as err:
            self.set_status(400)
            self.write(err.messages)
