# -*- encoding: utf-8 -*-
import logging
from typing import List, Union, Callable

from marshmallow import ValidationError
from retrying import retry

from travel.avia.travelers.application.handlers import BaseHandler
from travel.avia.travelers.application.models import Passenger, Document, BonusCard
from travel.avia.travelers.application.services.data_sync import data_sync_exceptions, DataSyncError
from travel.avia.travelers.application.schemas import CombinePassengersSchema, PassengerSchema

logger = logging.getLogger(__name__)
combine_passengers_schema = CombinePassengersSchema()
passenger_schema = PassengerSchema()

DocumentsType = Union[Document, BonusCard]


class CombinePassengersHandler(BaseHandler):
    def __init__(self, application, request, **kwargs):
        super(CombinePassengersHandler, self).__init__(application, request, **kwargs)
        self._check_user_ticket = True
        self._check_service_ticket = True

    @data_sync_exceptions(logger)
    def post(self, uid: str, passenger_id: str):
        self._check_user_uid(uid)
        passenger_to_update = self._get_passenger(uid, passenger_id)
        data = combine_passengers_schema.loads(self.request.body.decode('utf8'))

        try:
            passenger = self.data_sync_client.save_passenger(
                uid,
                self._user_ticket,
                passenger_to_update.fill(data),
            )
            self._combine_passengers(uid, passenger_to_update, data['passengers'])
            self.write(passenger_schema.dump(passenger))
        except ValueError:
            return self.send_error(400, reason='Wrong request')
        except ValidationError as err:
            self.set_status(400)
            return self.write(err.messages)

    def _combine_passengers(self, uid: str, main_passenger: Passenger, passenger_ids: List[str]):
        passenger_ids_to_delete = {p for p in passenger_ids if p != str(main_passenger.id)}
        if not passenger_ids_to_delete:
            return

        passengers_to_delete = [
            p for p in self._get_passengers(uid) if str(p.id) in passenger_ids_to_delete
        ]
        documents_to_migrate = [
            d for d in self.data_sync_client.get_documents(uid, self._user_ticket)
            if str(d.passenger_id) in passenger_ids_to_delete
        ]
        bonus_cards_to_migrate = [
            c for c in self.data_sync_client.get_bonus_cards(uid, self._user_ticket)
            if str(c.passenger_id) in passenger_ids_to_delete
        ]
        document_ids = [d.id for d in documents_to_migrate]
        bonus_card_ids = [c.id for c in bonus_cards_to_migrate]

        try:
            for d in documents_to_migrate:
                self._migrate_passenger_item(uid, main_passenger, d, self.data_sync_client.save_document)
            for c in bonus_cards_to_migrate:
                self._migrate_passenger_item(uid, main_passenger, c, self.data_sync_client.save_bonus_card)
            for p in passengers_to_delete:
                self.data_sync_client.delete_passenger(uid, self._user_ticket, p)
        except:
            logger.error(
                'Failed to combine passengers (%s) with documents (%s) and bonus cards (%s) into passenger "%s"',
                passenger_ids, document_ids, bonus_card_ids, main_passenger.id
            )
            raise

        logger.info(
            'Passengers (%s) with documents (%s) and bonus cards (%s) have combined into passenger "%s"',
            passenger_ids, document_ids, bonus_card_ids, main_passenger.id
        )

    @retry(
        stop_max_attempt_number=3,
        retry_on_exception=lambda exception: isinstance(exception, DataSyncError),
        wait_exponential_multiplier=100,
        wait_exponential_max=500
    )
    def _migrate_passenger_item(
        self, uid: str,
        target_passenger: Passenger,
        item: DocumentsType,
        save: Callable[[str, str, Passenger, DocumentsType], None]
    ):
        item.id = None
        save(uid, self._user_ticket, target_passenger, item)
