# -*- encoding: utf-8 -*-
import logging
from typing import Optional

from marshmallow import ValidationError

from travel.avia.travelers.application.handlers import BaseHandler
from travel.avia.travelers.application.models import Passenger, Document
from travel.avia.travelers.application.services.data_sync import data_sync_exceptions
from travel.avia.travelers.application.schemas import DocumentSchema


logger = logging.getLogger(__name__)
document_schema = DocumentSchema()
documents_schema = DocumentSchema(many=True)


class DocumentsHandler(BaseHandler):
    def __init__(self, application, request, **kwargs):
        super(DocumentsHandler, self).__init__(application, request, **kwargs)
        self._check_user_ticket = True
        self._check_service_ticket = True

    @data_sync_exceptions(logger)
    def get(self,  uid: str, passenger_id: str, document_id: Optional[str] = None):
        self._check_user_uid(uid)
        passenger = self._get_passenger(uid, passenger_id)

        if document_id is None:
            self.set_header('Content-Type', 'application/json')
            self.write(documents_schema.dumps(
                self.data_sync_client.get_passenger_documents(uid, self._user_ticket, passenger)
            ))
        else:
            self.write(document_schema.dump(
                self._get_document(uid, passenger, document_id)
            ))

    @data_sync_exceptions(logger)
    def post(self, uid: str, passenger_id: str, document_id: Optional[str] = None):
        self._check_user_uid(uid)
        if document_id:
            return self.send_error(405, reason='Use PUT for update document')
        self._update_document(uid, self._get_passenger(uid, passenger_id), Document())

    @data_sync_exceptions(logger)
    def put(self, uid: str, passenger_id: str, document_id: Optional[str] = None):
        self._check_user_uid(uid)
        if not document_id:
            return self.send_error(404, reason='Empty document id')

        passenger = self._get_passenger(uid, passenger_id)
        self._update_document(
            uid,
            passenger,
            self._get_document(uid, passenger, document_id),
        )

    @data_sync_exceptions(logger)
    def delete(self, uid: str, passenger_id: str, document_id: Optional[str] = None):
        self._check_user_uid(uid)
        passenger = self._get_passenger(uid, passenger_id)
        document = self._get_document(uid, passenger, document_id)

        self.data_sync_client.delete_document(uid, self._user_ticket, document)

        self.set_status(201, 'Deleted')

        self._yt_request_log(
            uid=uid,
            traveler_id=passenger.traveler_id,
            passenger_id=passenger.id,
            document_id=document.id,
        )

    def _update_document(self, uid: str, passenger: Passenger, document: Document):
        try:
            data = document_schema.loads(self.request.body.decode('utf8'))
            data['passenger_id'] = passenger.id

            document = self.data_sync_client.save_document(
                uid,
                self._user_ticket,
                passenger,
                document.fill(data),
            )

            self.write(document_schema.dump(document))

            self._yt_request_log(
                uid=uid,
                traveler_id=passenger.traveler_id,
                passenger_id=passenger.id,
                document_id=document.id,
            )

        except ValueError:
            return self.send_error(400, reason='Wrong request')
        except ValidationError as err:
            self.set_status(400)
            self.write(err.messages)
