# -*- encoding: utf-8 -*-
import logging
from typing import Optional

from marshmallow import ValidationError

from travel.avia.travelers.application.handlers import BaseHandler
from travel.avia.travelers.application.models import Traveler, Passenger
from travel.avia.travelers.application.services.data_sync import data_sync_exceptions
from travel.avia.travelers.application.schemas import PassengerSchema

logger = logging.getLogger(__name__)
passenger_schema = PassengerSchema()
passengers_schema = PassengerSchema(many=True)


class PassengersHandler(BaseHandler):
    def __init__(self, application, request, **kwargs):
        super(PassengersHandler, self).__init__(application, request, **kwargs)
        self._check_user_ticket = True
        self._check_service_ticket = True

    @data_sync_exceptions(logger)
    def get(self, uid: str, passenger_id: Optional[str] = None):
        self._check_user_uid(uid)
        self._get_traveler(uid)
        fields = self._get_fields()

        if passenger_id is None:
            self.set_header('Content-Type', 'application/json')
            passengers = self._get_passengers(uid, fields)
            self.write(passengers_schema.dumps(passengers))
        else:
            passenger = self._get_passenger(uid, passenger_id, fields)
            self.write(passenger_schema.dump(passenger))

    @data_sync_exceptions(logger)
    def post(self, uid: str, passenger_id: Optional[str] = None):
        self._check_user_uid(uid)
        if passenger_id:
            return self.send_error(405, reason='Use PUT for update passenger')
        self._update_passenger(uid, self._get_traveler(uid), Passenger())

    @data_sync_exceptions(logger)
    def put(self, uid: str, passenger_id: Optional[str] = None):
        self._check_user_uid(uid)
        if not passenger_id:
            return self.send_error(404, reason='Empty passenger id')

        self._update_passenger(
            uid,
            self._get_traveler(uid),
            self._get_passenger(uid, passenger_id),
        )

    @data_sync_exceptions(logger)
    def delete(self, uid: str, passenger_id: Optional[str] = None):
        self._check_user_uid(uid)
        self._get_traveler(uid)
        if not passenger_id:
            return self.send_error(404, reason='Empty passenger id')

        passenger = self._get_passenger(uid, passenger_id)

        self.data_sync_client.delete_passenger(uid, self._user_ticket, passenger)

        self.set_status(201, 'Deleted')

        self._yt_request_log(
            uid=uid,
            traveler_id=passenger.traveler_id,
            passenger_id=passenger.id,
        )

    def _update_passenger(self, uid: str, traveler: Traveler, passenger: Passenger):
        try:
            data = passenger_schema.loads(self.request.body.decode('utf8'))
            data['traveler_id'] = traveler.id

            passenger = self.data_sync_client.save_passenger(
                uid,
                self._user_ticket,
                passenger.fill(data),
            )

            self.write(passenger_schema.dump(passenger))

            self._yt_request_log(
                uid=uid,
                traveler_id=passenger.traveler_id,
                passenger_id=passenger.id,
            )

        except ValueError:
            return self.send_error(400, reason='Wrong request')
        except ValidationError as err:
            self.set_status(400)
            self.write(err.messages)
