# -*- encoding: utf-8 -*-
import logging

from marshmallow import ValidationError

from travel.avia.travelers.application.handlers import BaseHandler
from travel.avia.travelers.application.models import Traveler
from travel.avia.travelers.application.services.data_sync import data_sync_exceptions
from travel.avia.travelers.application.schemas import TravelerSchema

logger = logging.getLogger(__name__)
traveler_schema = TravelerSchema()


class TravelerHandler(BaseHandler):
    def __init__(self, application, request, **kwargs):
        super(TravelerHandler, self).__init__(application, request, **kwargs)
        self._check_user_ticket = True
        self._check_service_ticket = True

    @data_sync_exceptions(logger)
    def get(self, uid):
        self._check_user_uid(uid)
        fields = self._get_fields()
        traveler = self._get_traveler(uid)
        if 'passengers' in fields:
            traveler.fill({'passengers': self._get_passengers(uid, fields)})
        self.write(traveler_schema.dump(traveler))

    @data_sync_exceptions(logger)
    def post(self, uid):
        self._check_user_uid(uid)
        try:
            traveler = self._get_traveler(uid, silent=True)
            if not traveler:
                traveler = Traveler()

            data = traveler_schema.loads(self.request.body.decode('utf8'))

            traveler = self.data_sync_client.save_traveler(
                uid,
                self._user_ticket,
                traveler.fill(data),
            )

            self.write(traveler_schema.dump(traveler))

            self._yt_request_log(
                uid=uid,
                traveler_id=traveler.id if traveler else None,
            )

        except ValueError:
            return self.send_error(400, reason='Wrong request')
        except ValidationError as err:
            self.set_status(400)
            self.write(err.messages)
