# -*- encoding: utf-8 -*-
import logging
import time

from feature_flag_client import Client

from travel.avia.travelers.application import settings


_logger = logging.getLogger(__name__)


class FeatureFlagStorage(object):
    def __init__(self, feature_flag_client, logger=None, cache_ttl=settings.FEATURE_FLAG_CACHE_TTL):
        self._last_updated = time.time()
        self._cache_ttl = cache_ttl
        self._context = None
        self._logger = logger or _logger
        self._feature_flag_client = feature_flag_client  # type: Client

    @property
    def timer(self):
        return self._cache_ttl

    def reset_context(self):
        self._context = None

    def update_context_if_needed(self):
        now = time.time()
        if self._context is None or now - self._last_updated >= self._cache_ttl:
            self._logger.info('Updated feature flag context')
            self._context = self._feature_flag_client.create_context()
            self._last_updated = now

    @property
    def client(self):
        return self._feature_flag_client

    @client.setter
    def client(self, item):
        self._feature_flag_client = item
        self.reset_context()

    def flag_by_code(self, code):
        self.update_context_if_needed()
        return self._context.flag_enabled(code)

    @property
    def enable_itn_document(self):
        return self.flag_by_code('TR_ENABLE_ITN_DOCUMENT')


flag_storage = FeatureFlagStorage(
    Client(
        host=settings.FEATURE_FLAG_HOST,
        service_code='travelers',
        logger=logging.getLogger('feature_flag_client')),
    _logger,
)
