# -*- encoding: utf-8 -*-
import re
from datetime import date, datetime
from enum import Enum
from http_geobase.models import GeobaseRegion

from marshmallow import (
    Schema,
    fields,
    validates,
    validates_schema,
    ValidationError,
    EXCLUDE,
    pre_dump,
    post_dump,
    RAISE,
)
from marshmallow_enum import EnumField
from typing import Optional

from travel.avia.travelers.application.models import Passenger
from travel.avia.travelers.application.validations.document_types import DocumentType, DOCUMENT_SETTINGS
from travel.avia.travelers.application.validations.bonus_card_types import BonusCardType, BONUS_CARD_SETTINGS
from travel.avia.travelers.application.validations import validate_with_settings
from travel.avia.travelers.application.services.geodata.client import GeoDataClient

RE_PHONE = re.compile(r'^\+?\d{11,}$')


class Gender(Enum):
    male = 'male'
    female = 'female'


class StripedString(fields.String):
    def _deserialize(self, value, attr, data, **kwargs):
        return super(StripedString, self)._deserialize(value, attr, data, **kwargs).strip()


class PhoneSchemaMixin:
    @validates('phone')
    def phone_validate(self, value):
        self._validate_phone(value)

    @validates('phone_additional')
    def phone_additional_validate(self, value):
        self._validate_phone(value)

    @staticmethod
    def _validate_phone(value):
        if not value:
            return
        if not RE_PHONE.match(value):
            raise ValidationError('Wrong phone')


class DocumentSchema(Schema):
    class Meta:
        unknown = EXCLUDE

    id = fields.UUID(dump_only=True)
    passenger_id = fields.UUID(required=True, dump_only=True)
    type = EnumField(DocumentType, required=True, error='Possible values: {values}')
    title = StripedString(allow_none=True)
    number = StripedString(required=True)
    first_name = StripedString(allow_none=True)
    middle_name = StripedString(allow_none=True)
    last_name = StripedString(allow_none=True)
    first_name_en = StripedString(allow_none=True)
    middle_name_en = StripedString(allow_none=True)
    last_name_en = StripedString(allow_none=True)
    issue_date = fields.Date(format='%Y-%m-%d', allow_none=True)
    expiration_date = fields.Date(format='%Y-%m-%d', allow_none=True)
    citizenship = fields.Integer(allow_none=True)
    created_at = fields.DateTime(format='%Y-%m-%d %H:%M:%S', dump_only=True)
    updated_at = fields.DateTime(format='%Y-%m-%d %H:%M:%S', dump_only=True)

    @validates('issue_date')
    def validate_issue_date(self, value):
        if value and value.strftime('%Y-%m-%d') > datetime.now().strftime('%Y-%m-%d'):
            raise ValidationError('Issue date in the future')

    @validates('expiration_date')
    def validate_expiration_date(self, value):
        if value and value.strftime('%Y-%m-%d') < datetime.now().strftime('%Y-%m-%d'):
            raise ValidationError('Expiration date in the past')

    @validates('citizenship')
    def validate_citizenship(self, value):
        geo_data_client = GeoDataClient.get_instance()
        if value is not None and not geo_data_client.country_by_geo_id(value):
            raise ValidationError('Citizenship is not valid country')

    @validates_schema
    def validate_document(self, data, **kwargs):
        if data['type'] not in DOCUMENT_SETTINGS:
            return
        validation_settings = DOCUMENT_SETTINGS[data['type']]
        validate_with_settings(data, validation_settings)

        document_country = self._get_document_country(data['type'])
        if data['type'] != DocumentType.other and document_country and 'citizenship' in data:
            if data['citizenship'] != document_country.id:
                raise ValidationError(
                    'Citizenship country for {type} is not {country}'.format(
                        type=data['type'], country=document_country.iso_name
                    )
                )

    @pre_dump
    def unify_citizenship(self, document, **kwargs):
        if document.citizenship:
            try:
                document.citizenship = int(document.citizenship)
                if document.citizenship == 0:
                    document.citizenship = None
            except ValueError:
                geo_data_client = GeoDataClient.get_instance()
                country = geo_data_client.country_by_iso_name(document.citizenship)
                document.citizenship = int(country.id) if country else None

        if document.type:
            document_country = self._get_document_country(document.type)
            if not document.citizenship and document_country and document.type != DocumentType.other:
                document.citizenship = int(document_country.id)

        return document

    @staticmethod
    def _get_document_country(document_type: DocumentType) -> Optional[GeobaseRegion]:
        if not document_type:
            return None

        country_prefix, _, rest = document_type.name.partition('_')
        if not country_prefix:
            return None

        geo_data_client = GeoDataClient.get_instance()
        return geo_data_client.country_by_iso_name(country_prefix.upper())


class BonusCardSchema(Schema):
    class Meta:
        unknown = EXCLUDE

    id = fields.UUID(dump_only=True)
    passenger_id = fields.UUID(required=True, dump_only=True)
    type = EnumField(BonusCardType, required=True, error='Possible values: {values}')
    title = StripedString()
    number = StripedString()
    company_id = fields.Integer()
    created_at = fields.DateTime(format='%Y-%m-%d %H:%M:%S', dump_only=True)
    updated_at = fields.DateTime(format='%Y-%m-%d %H:%M:%S', dump_only=True)

    @validates_schema
    def validate_bonus_card(self, data, **kwargs):
        if data['type'] not in BONUS_CARD_SETTINGS:
            return
        validation_settings = BONUS_CARD_SETTINGS[data['type']]
        validate_with_settings(data, validation_settings)


class DocumentTypeSettingsSchema(Schema):
    class Meta:
        unknown = EXCLUDE

    required = fields.List(fields.String(), dump_only=True)
    unused = fields.List(fields.String(), dump_only=True)
    re_validations = fields.Dict(keys=fields.String(), values=fields.String(), dump_only=True)


class PassengerSchema(Schema, PhoneSchemaMixin):
    class Meta:
        unknown = EXCLUDE

    id = fields.UUID(dump_only=True)
    title = StripedString(required=True)
    gender = EnumField(Gender, allow_none=True, error='Possible values: {values}')
    birth_date = fields.Date(format='%Y-%m-%d', allow_none=True)
    phone = StripedString(allow_none=True)
    phone_additional = StripedString(allow_none=True)
    email = fields.Email(allow_none=True)
    itn = StripedString(allow_none=True)
    train_notifications_enabled = fields.Boolean(allow_none=True)
    created_at = fields.DateTime(format='%Y-%m-%d %H:%M:%S', dump_only=True)
    updated_at = fields.DateTime(format='%Y-%m-%d %H:%M:%S', dump_only=True)
    documents = fields.Nested(DocumentSchema, many=True, dump_only=True)
    bonus_cards = fields.Nested(BonusCardSchema, many=True, dump_only=True)

    @validates('birth_date')
    def validate_birth_date(self, value: date):
        if value and value.strftime('%Y-%m-%d') > datetime.now().strftime('%Y-%m-%d'):
            raise ValidationError('Birth date in the future')

    @validates('itn')
    def validate_itn(self, value: str):
        if value and not re.match(r'^\d{12}$', value):
            raise ValidationError('ITN must contain 12 digits.')

    @post_dump
    def delete_none_nested_fields(self, data, **kwargs):
        if 'documents' in data and data['documents'] is None:
            del data['documents']
        if 'bonus_cards' in data and data['bonus_cards'] is None:
            del data['bonus_cards']
        return data

    def get_attribute(self, obj, attr, default):
        if attr in Passenger.ATTRIBUTES:
            return obj.attributes[attr]

        return super(PassengerSchema, self).get_attribute(obj, attr, default)


class TravelerSchema(Schema, PhoneSchemaMixin):
    class Meta:
        unknown = EXCLUDE

    agree = fields.Boolean(required=True)
    phone = StripedString()
    phone_additional = StripedString(allow_none=True)
    email = fields.Email(allow_none=True)
    created_at = fields.DateTime(format='%Y-%m-%d %H:%M:%S', dump_only=True)
    updated_at = fields.DateTime(format='%Y-%m-%d %H:%M:%S', dump_only=True)
    passengers = fields.Nested(PassengerSchema, many=True, dump_only=True)

    @post_dump
    def delete_none_passengers(self, data, **kwargs):
        if 'passengers' in data and data['passengers'] is None:
            del data['passengers']
        return data


class CombinePassengersSchema(Schema, PhoneSchemaMixin):
    class Meta:
        unknown = RAISE

    title = StripedString()
    gender = EnumField(Gender, error='Possible values: {values}')
    birth_date = fields.Date(format='%Y-%m-%d')
    phone = StripedString(allow_none=True)
    phone_additional = StripedString(allow_none=True)
    email = fields.Email(allow_none=True)
    passengers = fields.List(fields.String(), required=True)
