# -*- encoding: utf-8 -*-
import logging
from datetime import datetime
from functools import partial
from typing import List, Optional

from requests import Session, HTTPError, Response
from requests.packages.urllib3.util.retry import Retry
from requests.adapters import HTTPAdapter
from retrying import retry

from travel.library.python.tvm_ticket_provider import provider_fabric

from travel.avia.travelers.application import settings
from travel.avia.travelers.application.services.data_sync import schemas, ForbiddenDataSyncError, DataSyncError
from travel.avia.travelers.application.models import Traveler, Passenger, Document, BonusCard  # noqa

logger = logging.getLogger(__name__)


class DataSyncClient(object):
    def __init__(self, api_url, tvm_id, tvm_client, request_timeout=0.5):
        self._api_url = api_url
        self._tvm_id = tvm_id
        self._tvm_client = tvm_client
        self._traveler_schema = schemas.TravelerSchema()
        self._passenger_schema = schemas.PassengerSchema()
        self._passengers_schema = schemas.PassengerSchema(many=True)
        self._document_schema = schemas.DocumentSchema()
        self._documents_schema = schemas.DocumentSchema(many=True)
        self._bonus_card_schema = schemas.BonusCardSchema()
        self._bonus_cards_schema = schemas.BonusCardSchema(many=True)

        self._request_timeout = request_timeout

        self._session = Session()
        adapter = HTTPAdapter(
            max_retries=Retry(
                total=3,
                read=3,
                connect=3,
                backoff_factor=0.1,
                status_forcelist=(500, 502, 503, 504),
                method_whitelist=('GET', 'PUT', 'DELETE'),
            ),
        )
        self._session.mount('http://', adapter)
        self._session.mount('https://', adapter)

        self._post_session = Session()
        adapter = HTTPAdapter(
            max_retries=Retry(
                total=3,
                read=3,
                connect=3,
                backoff_factor=0.1,
                status_forcelist=(500,),
                method_whitelist=('POST',),
            ),
        )
        self._post_session.mount('http://', adapter)
        self._post_session.mount('https://', adapter)

    def get_traveler(self, uid: str, user_ticket: str) -> Optional[Traveler]:
        response = self._get('/traveler', uid=uid, user_ticket=user_ticket)
        if response is None:
            return None
        items = response.get('items', [])

        return self._traveler_schema.load(items[0]) if len(items) > 0 else None

    def save_traveler(self, uid: str, user_ticket: str, traveler: Traveler) -> Traveler:
        traveler.updated_at = datetime.now()
        if traveler.id:
            response = self._put(
                '/traveler/{}'.format(traveler.id),
                data=self._traveler_schema.dump(traveler),
                uid=uid,
                user_ticket=user_ticket,
            )
        else:
            traveler.created_at = datetime.now()
            response = self._post(
                '/traveler',
                data=self._traveler_schema.dump(traveler),
                uid=uid,
                user_ticket=user_ticket,
            )
            self._check_exist(partial(self.get_traveler, uid, user_ticket))

        return self._traveler_schema.load(response)

    def get_passenger(self, uid: str, user_ticket: str, passenger_id: str) -> Optional[Passenger]:
        passengers = self.get_passengers(uid, user_ticket)
        for passenger in passengers:
            if str(passenger.id) == passenger_id:
                return passenger

        return None

    def get_passengers(self, uid: str, user_ticket: str) -> List[Passenger]:
        response = self._get(
            '/traveler/passengers',
            uid=uid,
            user_ticket=user_ticket,
        )

        return self._passengers_schema.load(response.get('items', []))

    def save_passenger(self, uid: str, user_ticket: str, passenger: Passenger) -> Passenger:
        passenger.updated_at = datetime.now()

        if passenger.id:
            response = self._put(
                '/traveler/passengers/' + str(passenger.id),
                data=self._passenger_schema.dump(passenger),
                uid=uid,
                user_ticket=user_ticket,
            )
        else:
            passenger.created_at = datetime.now()
            response = self._post(
                '/traveler/passengers',
                data=self._passenger_schema.dump(passenger),
                uid=uid,
                user_ticket=user_ticket,
            )
            self._check_exist(partial(self.get_passenger, uid, user_ticket, response['id']))

        return self._passenger_schema.load(response)

    def delete_passenger(self, uid: str, user_ticket: str, passenger: Passenger) -> bool:
        for document in self.get_passenger_documents(uid, user_ticket, passenger):
            self.delete_document(uid, user_ticket, document)

        for bonus_card in self.get_passenger_bonus_cards(uid, user_ticket, passenger):
            self.delete_bonus_card(uid, user_ticket, bonus_card)

        self._delete(
            '/traveler/passengers/' + str(passenger.id),
            uid=uid,
            user_ticket=user_ticket,
        )

        return True

    def get_document(self, uid: str, user_ticket: str, passenger: Passenger, document_id: str) -> Optional[Document]:
        documents = self.get_passenger_documents(uid, user_ticket, passenger)
        for document in documents:
            if str(document.id) == document_id:
                return document

        return None

    def get_documents(self, uid: str, user_ticket: str) -> List[Document]:
        response = self._get(
            '/traveler/passengers/documents',
            uid=uid,
            user_ticket=user_ticket,
        )
        return self._documents_schema.load(response.get('items', []))

    def get_passenger_documents(self, uid: str, user_ticket: str, passenger: Passenger) -> List[Document]:
        return [d for d in self.get_documents(uid, user_ticket) if d.passenger_id == passenger.id]

    def save_document(self, uid: str, user_ticket: str, passenger: Passenger, document: Document) -> Document:
        document.passenger_id = passenger.id
        document.updated_at = datetime.now()

        if document.id:
            response = self._put(
                '/traveler/passengers/documents/' + str(document.id),
                data=self._document_schema.dump(document),
                uid=uid,
                user_ticket=user_ticket,
            )
        else:
            document.created_at = datetime.now()
            response = self._post(
                '/traveler/passengers/documents',
                data=self._document_schema.dump(document),
                uid=uid,
                user_ticket=user_ticket,
            )
            self._check_exist(partial(self.get_document, uid, user_ticket, passenger, response['id']))

        return self._document_schema.load(response)

    def delete_document(self, uid: str, user_ticket: str, document: Document) -> bool:
        self._delete(
            '/traveler/passengers/documents/' + str(document.id),
            uid=uid,
            user_ticket=user_ticket,
        )

        return True

    def get_bonus_card(self, uid: str, user_ticket: str, passenger: Passenger, bonus_card_id: str) -> Optional[BonusCard]:
        bonus_cards = self.get_passenger_bonus_cards(uid, user_ticket, passenger)
        for bonus_card in bonus_cards:
            if str(bonus_card.id) == bonus_card_id:
                return bonus_card

        return None

    def get_bonus_cards(self, uid: str, user_ticket: str) -> List[BonusCard]:
        response = self._get(
            '/traveler/passengers/bonus_cards',
            uid=uid,
            user_ticket=user_ticket,
        )
        return self._bonus_cards_schema.load(response.get('items', []))

    def get_passenger_bonus_cards(self, uid: str, user_ticket: str, passenger: Passenger) -> List[BonusCard]:
        return [c for c in self.get_bonus_cards(uid, user_ticket) if c.passenger_id == passenger.id]

    def save_bonus_card(self, uid: str, user_ticket: str, passenger: Passenger, bonus_card: BonusCard) -> BonusCard:
        bonus_card.passenger_id = passenger.id
        bonus_card.updated_at = datetime.now()

        if bonus_card.id:
            response = self._put(
                '/traveler/passengers/bonus_cards/' + str(bonus_card.id),
                data=self._bonus_card_schema.dump(bonus_card),
                uid=uid,
                user_ticket=user_ticket,
            )
        else:
            bonus_card.created_at = datetime.now()
            response = self._post(
                '/traveler/passengers/bonus_cards',
                data=self._bonus_card_schema.dump(bonus_card),
                uid=uid,
                user_ticket=user_ticket,
            )
            self._check_exist(partial(self.get_bonus_card, uid, user_ticket, passenger, response['id']))

        return self._bonus_card_schema.load(response)

    def delete_bonus_card(self, uid: str, user_ticket: str, bonus_card: BonusCard) -> bool:
        self._delete(
            '/traveler/passengers/bonus_cards/' + str(bonus_card.id),
            uid=uid,
            user_ticket=user_ticket,
        )

        return True

    def _get(self, path, uid, user_ticket):
        service_ticket = self._tvm_client.get_ticket(self._tvm_id)

        response = self._session.get(
            self._api_url + path,
            headers={
                'X-Uid': uid,
                'X-Ya-Service-Ticket': service_ticket,
                'X-Ya-User-Ticket': user_ticket,
            },
            timeout=self._request_timeout,
        )

        if response.status_code == 404:
            return None

        self._raise_for_status(response)

        return response.json()

    def _post(self, path, data, uid, user_ticket):
        service_ticket = self._tvm_client.get_ticket(self._tvm_id)

        response = self._post_session.post(
            self._api_url + path,
            headers={
                'X-Uid': uid,
                'X-Ya-Service-Ticket': service_ticket,
                'X-Ya-User-Ticket': user_ticket,
                'Content-Type': 'application/json',
            },
            json=data,
        )

        self._raise_for_status(response)

        return response.json()

    def _put(self, path, data, uid, user_ticket):
        service_ticket = self._tvm_client.get_ticket(self._tvm_id)

        response = self._session.put(
            self._api_url + path,
            headers={
                'X-Uid': uid,
                'X-Ya-Service-Ticket': service_ticket,
                'X-Ya-User-Ticket': user_ticket,
            },
            json=data,
            timeout=self._request_timeout,
        )

        self._raise_for_status(response)

        return response.json()

    def _delete(self, path, uid, user_ticket):
        service_ticket = self._tvm_client.get_ticket(self._tvm_id)

        response = self._session.delete(
            self._api_url + path,
            headers={
                'X-Uid': uid,
                'X-Ya-Service-Ticket': service_ticket,
                'X-Ya-User-Ticket': user_ticket,
            },
            timeout=self._request_timeout,
        )

        self._raise_for_status(response)

        return response.json()

    @staticmethod
    def _raise_for_status(response: Response):
        try:
            response.raise_for_status()
        except HTTPError:
            if response.status_code == 403:
                raise ForbiddenDataSyncError()

            raise DataSyncError('%s DataSync error: %s for %s\nResponse:\n%s' % (
                response.status_code,
                response.reason,
                response.url,
                response.content,
            ))

    @staticmethod
    @retry(
        stop_max_attempt_number=5,
        wait_fixed=500,
        retry_on_exception=lambda exception: isinstance(exception, RuntimeError) and str(exception) == 'Not exist',
    )
    def _check_exist(getter):
        d = getter()
        if d is None:
            logger.warning('Not found in check for %s', getter)
            raise RuntimeError('Not exist')


def create_datasync():
    return DataSyncClient(
        settings.DATA_SYNC_API_URL,
        settings.DATA_SYNC_TVM_SERVICE_ID,
        provider_fabric.create(
            source_id=settings.TVM_SERVICE_ID,
            secret=settings.TVM_SECRET,
            destinations=[settings.DATA_SYNC_TVM_SERVICE_ID],
        ),
    )
