# -*- encoding: utf-8 -*-
import copy

from marshmallow import Schema, fields, post_load, post_dump
from marshmallow_enum import EnumField

from travel.avia.travelers.application.models import Traveler, Passenger, Document, BonusCard
from travel.avia.travelers.application.schemas import Gender, DocumentType, BonusCardType


class BaseDataSyncSchema(Schema):
    @post_dump
    def null_to_str(self, data, **kwargs):
        result = dict()
        for k in data:
            if data[k] is not None:
                result[k] = data[k]

        return result


class TravelerSchema(BaseDataSyncSchema):
    traveler_id = fields.UUID(load_only=True, data_key='id', attribute='id')
    agree = fields.Boolean()
    phone = fields.String()
    phone_additional = fields.String()
    email = fields.String()
    created_at = fields.DateTime(format='%Y-%m-%dT%H:%M:%S.000Z')
    updated_at = fields.DateTime(format='%Y-%m-%dT%H:%M:%S.000Z')

    @post_load
    def make_traveler(self, data, **kwargs):
        return Traveler().fill(data)


class PassengerSchema(BaseDataSyncSchema):
    passenger_id = fields.UUID(load_only=True, data_key='id', attribute='id')
    traveler_id = fields.UUID()
    title = fields.String()
    gender = EnumField(Gender)
    birth_date = fields.DateTime(format='%Y-%m-%dT%H:%M:%S.000Z')
    phone = fields.String()
    phone_additional = fields.String()
    itn = fields.String()
    email = fields.String()
    attributes = fields.List(fields.String())
    created_at = fields.DateTime(format='%Y-%m-%dT%H:%M:%S.000Z')
    updated_at = fields.DateTime(format='%Y-%m-%dT%H:%M:%S.000Z')

    @post_load
    def make_passenger(self, data: dict, **kwargs) -> Passenger:
        d = copy.deepcopy(data)
        if 'attributes' in d:
            d['attributes'] = {a: a in d['attributes'] for a in Passenger.ATTRIBUTES}

        return Passenger().fill(d)

    def get_attribute(self, obj, attr, default):
        if attr == 'attributes':
            return {a for a in Passenger.ATTRIBUTES if obj.attributes[a]}

        return super(PassengerSchema, self).get_attribute(obj, attr, default)


class DocumentSchema(BaseDataSyncSchema):
    document_id = fields.UUID(load_only=True, data_key='id', attribute='id')
    passenger_id = fields.UUID()
    type = EnumField(DocumentType)
    title = fields.String()
    number = fields.String()
    first_name = fields.String()
    middle_name = fields.String()
    last_name = fields.String()
    first_name_en = fields.String()
    middle_name_en = fields.String()
    last_name_en = fields.String()
    issue_date = fields.DateTime(format='%Y-%m-%dT%H:%M:%S.000Z')
    expiration_date = fields.DateTime(format='%Y-%m-%dT%H:%M:%S.000Z')
    citizenship = fields.String()
    created_at = fields.DateTime(format='%Y-%m-%dT%H:%M:%S.000Z')
    updated_at = fields.DateTime(format='%Y-%m-%dT%H:%M:%S.000Z')

    @post_load
    def make_document(self, data, **kwargs):
        return Document().fill(data)


class BonusCardSchema(BaseDataSyncSchema):
    bonus_card_id = fields.UUID(load_only=True, data_key='id', attribute='id')
    passenger_id = fields.UUID()
    type = EnumField(BonusCardType)
    title = fields.String()
    number = fields.String()
    company_id = fields.Integer()
    created_at = fields.DateTime(format='%Y-%m-%dT%H:%M:%S.000Z')
    updated_at = fields.DateTime(format='%Y-%m-%dT%H:%M:%S.000Z')

    @post_load
    def make_bonus_card(self, data, **kwargs):
        return BonusCard().fill(data)
