package main

import (
	"context"
	"fmt"
	"sync"
	"time"

	ghealth "google.golang.org/grpc/health"
	healthpb "google.golang.org/grpc/health/grpc_health_v1"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/maxprocs"

	"a.yandex-team.ru/travel/avia/weekendtour/internal/app"
	"a.yandex-team.ru/travel/avia/weekendtour/internal/database"
	handler "a.yandex-team.ru/travel/avia/weekendtour/internal/handlers"
	"a.yandex-team.ru/travel/avia/weekendtour/internal/models"
	"a.yandex-team.ru/travel/avia/weekendtour/internal/service/consumers"
	"a.yandex-team.ru/travel/avia/weekendtour/internal/service/polling"
	"a.yandex-team.ru/travel/avia/weekendtour/internal/service/providers/tours"
	"a.yandex-team.ru/travel/library/go/configuration"
	grpcserver "a.yandex-team.ru/travel/library/go/grpcutil/server"
	httpserver "a.yandex-team.ru/travel/library/go/httputil/server"
	"a.yandex-team.ru/travel/library/go/logging"
	"a.yandex-team.ru/travel/library/go/metrics"
	metricserver "a.yandex-team.ru/travel/library/go/metrics/server"
	"a.yandex-team.ru/travel/library/go/tracing"
	"a.yandex-team.ru/travel/library/go/vault"
)

const serviceName = "weekendtour"

func main() {
	maxprocs.AdjustAuto()

	// setting up infrastructure
	ctx, ctxCancel := context.WithCancel(context.Background())
	config := configuration.NewDefaultConfitaLoader()
	err := config.Load(ctx, &app.Cfg)

	if err != nil {
		fmt.Println("can not load configuration:", err)
		ctxCancel()
		return
	}

	logger, err := logging.New(&app.Cfg.Logging)
	if err != nil {
		fmt.Println("failed to create logger, err:", err)
		ctxCancel()
		return
	}

	tracerCloser := tracing.InitializeDefaultTracer(serviceName)
	defer func() {
		err = tracerCloser.Close()
		if err != nil {
			logger.Error("tracer close error:", log.Error(err))
		}

		err = logger.L.Sync()
		if err != nil {
			fmt.Println("failed to close logger:", err)
			return
		}
		ctxCancel()
	}()

	rootRegistry := metrics.NewRegistryWithDeployTagsAndExplicitHost()
	// in order to make it possible to collect arbitrary application metrics through metrics.GlobalAppMetrics
	appMetrics := metrics.NewAppMetrics(rootRegistry.WithPrefix("app"))
	metrics.SetGlobalAppMetrics(appMetrics)
	metrics.RunPerfMetricsUpdater(rootRegistry, app.Cfg.Metrics.PerfMetricsRefreshInterval)

	pgClient, err := app.BuildPGClient(logger)
	if err != nil {
		logger.Error("failed to create PG client", log.Error(err))
		return
	}

	if err := models.MigrateAndInit(pgClient); err != nil {
		logger.Errorf(err.Error())
		return
	}

	grpcServiceRegisterers := []grpcserver.ServiceRegisterer{}

	weekendTourService := tours.NewService(logger)
	httpRouteBuilders := []httpserver.RouteBuilder{
		handler.NewHTTPWeekendTourHandler(weekendTourService, logger).GetRouteBuilder(),
	}

	healthServer := ghealth.NewServer()
	// The health-checking logic of a specific application should be implemented below
	go func() {
		for range time.Tick(2 * time.Second) {
			healthServer.SetServingStatus(serviceName, healthpb.HealthCheckResponse_SERVING)
		}
	}()

	// Start polling ticket-daemon variants logs
	resolver := vault.NewYavSecretsResolverFromToken(app.Cfg.YavToken)
	repository := database.NewDBRepository(pgClient, false)
	consumer, err := consumers.NewRoundTripConsumer(context.Background(), repository, logger)
	if err != nil {
		logger.Fatal("Unable to create round-trips consumer", log.Error(err))
	}

	service := polling.NewFlightVariantsPollingService(app.Cfg.YTReader, resolver, consumer.Consume, logger)
	err = service.Start(context.Background())
	if err != nil {
		logger.Fatal("Unable to start polling ticket-daemon variants logs", log.Error(err))
	}

	// running server
	wg := sync.WaitGroup{}
	wg.Add(3)

	go func() {
		defer wg.Done()
		server := grpcserver.NewDefaultGrpcServerBuilder(
			app.Cfg.Grpc,
			grpcServiceRegisterers,
			logger,
			nil, // no TVM for now
			nil, // no TVM-allowed IDs
			rootRegistry.WithPrefix("grpc"),
		).WithHealthServer(healthServer).Build()
		err = server.Run(context.Background())
		if err != nil {
			logger.Fatal("Error while starting Grpc server", log.Error(err))
		}
	}()

	go func() {
		defer wg.Done()
		err = metricserver.RunMetricsHTTPServer(context.Background(), app.Cfg.Metrics, logger, rootRegistry)
		if err != nil {
			logger.Fatal("Error while starting metrics server", log.Error(err))
		}
	}()

	go func() {
		defer wg.Done()
		err = httpserver.RunHTTPServer(context.Background(), app.Cfg.HTTP, httpRouteBuilders, logger, nil, nil, rootRegistry.WithPrefix("http"))
		if err != nil {
			logger.Fatal("Error while starting HTTP server", log.Error(err))
		}
	}()

	logger.Info("Started")
	wg.Wait()
}
