package database

import (
	"context"

	"golang.yandex/hasql"
	"gorm.io/gorm"
	"gorm.io/gorm/clause"

	"a.yandex-team.ru/travel/avia/weekendtour/internal/models"
	"a.yandex-team.ru/travel/avia/weekendtour/internal/pgclient"
)

type DBRepository struct {
	pgClient *pgclient.PGClient
	debug    bool
}

func NewDBRepository(pgClient *pgclient.PGClient, debug bool) *DBRepository {
	return &DBRepository{pgClient: pgClient, debug: debug}
}

func (r *DBRepository) CreateTour(ctx context.Context, tour models.WeekendTour) (*models.WeekendTour, error) {
	err := r.pgClient.ExecuteInTransaction(
		hasql.Primary,
		func(db *gorm.DB) error {
			db = db.WithContext(ctx)
			return db.Create(&tour).Error
		},
	)
	if err != nil {
		return nil, err
	}
	return &tour, nil
}

func (r *DBRepository) StoreRoundTrip(ctx context.Context, roundTrip models.RoundTrip) (*models.RoundTrip, error) {
	err := r.pgClient.ExecuteInTransaction(
		hasql.Primary,
		func(db *gorm.DB) error {
			db = db.WithContext(ctx)
			return db.Clauses(clause.OnConflict{DoNothing: true}).Create(&roundTrip).Error
		},
	)
	if err != nil {
		return nil, err
	}
	return &roundTrip, nil
}

func (r *DBRepository) GetRoundTrips(ctx context.Context, startingFromWeekend uint) ([]models.RoundTrip, error) {
	roundTrips := make([]models.RoundTrip, 0)
	err := r.pgClient.ExecuteInTransaction(
		hasql.Primary,
		func(db *gorm.DB) error {
			if r.debug {
				db = db.Debug()
			}
			return db.
				WithContext(ctx).
				Preload(clause.Associations).
				Model(&models.RoundTrip{}).
				Where("weekend_id >= ?", startingFromWeekend).
				Find(&roundTrips).
				Error
		},
	)
	return roundTrips, err
}

func (r *DBRepository) DeleteAll(ctx context.Context) error {
	err := r.pgClient.ExecuteInTransaction(
		hasql.Primary,
		func(db *gorm.DB) error {
			db = db.WithContext(ctx)
			db = db.Session(&gorm.Session{Context: ctx, AllowGlobalUpdate: true})
			return db.Delete(&models.WeekendTour{}).Error
		},
	)
	return err
}
