package handler

import (
	"context"
	"encoding/json"
	"fmt"
	"golang.org/x/xerrors"
	"net/http"
	"net/url"
	"strconv"
	"time"

	"github.com/go-chi/chi/v5"
	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/weekendtour/internal/models"
	"a.yandex-team.ru/travel/avia/weekendtour/internal/service/utils"
	"a.yandex-team.ru/travel/library/go/httputil"
)

type WeekendTourService interface {
	GetTours(ctx context.Context, request models.WeekendTourRequest) (response models.WeekendTourResponse, err error)
}

type HTTPWeekendTourHandler struct {
	weekendTourService WeekendTourService
	logger             log.Logger
}

func NewHTTPWeekendTourHandler(
	service WeekendTourService,
	l log.Logger,
) *HTTPWeekendTourHandler {
	return &HTTPWeekendTourHandler{
		weekendTourService: service,
		logger:             l,
	}
}

func (h *HTTPWeekendTourHandler) GetWeekendTours(w http.ResponseWriter, r *http.Request) {
	span, ctx := opentracing.StartSpanFromContext(r.Context(), "internal.handler.HTTPWeekendTourHandler:GetWeekendTours")
	defer span.Finish()

	params, err := url.ParseQuery(r.URL.RawQuery)
	if err != nil {
		httputil.HandleError(err, http.StatusBadRequest, w)
		return
	}
	from := params.Get("from")
	if from == "" {
		from = "MOW"
	}
	weekendID, err := strconv.Atoi(params.Get("weekendId"))
	if err != nil {
		httputil.HandleError(err, http.StatusBadRequest, w)
		return
	}
	weekday, err := utils.GetWeekdayForFlight(params.Get("weekendId"))
	if err != nil {
		httputil.HandleError(err, http.StatusBadRequest, w)
		return
	}
	if weekday != time.Friday {
		httputil.HandleError(xerrors.Errorf("weekendID has to be Friday"), http.StatusBadRequest, w)
		return
	}

	query := models.NewWeekendTourRequest(from, int32(weekendID))
	result, err := h.weekendTourService.GetTours(ctx, query)

	if err != nil {
		h.logger.Error("Internal error", log.Error(err))
		httputil.HandleError(fmt.Errorf("internal error"), http.StatusInternalServerError, w)
		return
	}

	resultBytes, err := json.Marshal(result)
	if err != nil {
		httputil.HandleError(err, http.StatusInternalServerError, w)
		return
	}
	w.Header().Set("Content-Type", "application/json")
	w.WriteHeader(http.StatusOK)
	_, _ = w.Write(resultBytes)
	_, _ = w.Write([]byte("\n"))
}

func (h *HTTPWeekendTourHandler) GetRouteBuilder() func(r chi.Router) {
	return func(r chi.Router) {
		r.Get("/weekend-tours", h.GetWeekendTours)
	}
}
