package slices

import (
	"a.yandex-team.ru/library/go/core/xerrors"
)

type Selector func(current, next interface{}) bool

type SelectorInt func(current, next int) bool

type SelectorString func(current, next string) bool

var (
	MinSelectorInt = SelectorInt(func(current, next int) bool {
		return next < current
	})

	MaxSelectorInt = SelectorInt(func(current, next int) bool {
		return next > current
	})

	MinSelectorString = SelectorString(func(current, next string) bool {
		return next < current
	})

	MaxSelectorString = SelectorString(func(current, next string) bool {
		return next > current
	})

	EmptySliceError = xerrors.New("unable to select from the empty slice")
)

func SelectInt(data []int, selector SelectorInt) (int, error) {
	if len(data) == 0 {
		return 0, EmptySliceError
	}
	current := 0
	for i, v := range data {
		if i == 0 || selector(current, v) {
			current = v
		}
	}
	return current, nil
}

func SelectString(data []string, selector SelectorString) (string, error) {
	if len(data) == 0 {
		return "", EmptySliceError
	}
	current := ""
	for i, v := range data {
		if i == 0 || selector(current, v) {
			current = v
		}
	}
	return current, nil
}

func Select(data []interface{}, selector Selector) (interface{}, error) {
	if len(data) == 0 {
		return nil, EmptySliceError
	}
	var current interface{}
	for i, v := range data {
		if i == 0 || selector(current, v) {
			current = v
		}
	}
	return current, nil
}
