package ytreader

import (
	"context"
	"fmt"
	"runtime"
	"strings"

	"golang.org/x/text/language"
	"golang.org/x/text/message"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/avia/weekendtour/internal/library/slices"
	vault "a.yandex-team.ru/travel/library/go/vault"
	"a.yandex-team.ru/yt/go/ypath"
	"a.yandex-team.ru/yt/go/yt"
	"a.yandex-team.ru/yt/go/yt/ythttp"
)

type Config struct {
	Proxy                string `config:"YT_PROXY" yaml:"proxy"`
	Token                string `config:"YT_TOKEN" yaml:"yt_token"`
	TablePath            string `config:"YT_TABLE_PATH" yaml:"yt_table_path"`
	PollingInterval      int32  `config:"YT_POLLING_INTERVAL" yaml:"yt_polling_interval"` // minutes; set to 0 to disable
	MaxMemoryConsumption int32  `config:"YT_MAX_MEMORY" yaml:"yt_max_memory"`             // in GBytes
}

var (
	DefaultConfig = Config{
		Proxy:                "hahn",
		Token:                "sec-01dfxmszhq27tk66hm107d0ycd",
		TablePath:            "//logs/avia-ticket-daemon-api-variants-log/30min",
		PollingInterval:      30,
		MaxMemoryConsumption: 4,
	}
)

type YTReader struct {
	ytClient yt.Client
	config   Config
	logger   log.Logger
}

type TDVariants struct {
	QID      string    `yson:"qid"`
	Variants []Variant `yson:"variants"`
}

type JourneyBoundaries struct {
	PointFromKey  string
	PointToKey    string
	DepartureDate string
	ReturnDate    string
}

func (vs *TDVariants) ParseQID() (JourneyBoundaries, error) {
	parts := strings.Split(vs.QID, ".")
	if len(parts) < 4 {
		return JourneyBoundaries{}, xerrors.New("qid has less than 4 parts")
	}
	parts = strings.Split(parts[3], "_")
	if len(parts) < 4 {
		return JourneyBoundaries{}, xerrors.New("qid has less than 4 params for journey boundaries")
	}
	result := JourneyBoundaries{
		PointFromKey:  parts[0],
		PointToKey:    parts[1],
		DepartureDate: parts[2],
		ReturnDate:    parts[3],
	}
	if strings.ToLower(result.ReturnDate) == "none" {
		result.ReturnDate = ""
	}
	return result, nil
}

type Variant struct {
	Partner string     `yson:"partner" json:"partner"`
	Route   [][]string `yson:"route" json:"route"`
	Price   Price      `yson:"tariff" json:"tariff"`
}

type Price struct {
	Currency string  `yson:"currency" json:"currency"`
	Value    float64 `yson:"value" json:"value"`
}

type VariantsConsumer func(ctx context.Context, v TDVariants) error

func NewYTReader(config Config, resolver *vault.YavSecretsResolver, logger log.Logger) (*YTReader, error) {
	token, err := resolver.GetSecretValue(config.Token, "token")
	if err != nil {
		return nil, err
	}
	ytConfig := &yt.Config{
		Logger: logger.Structured(),
		Proxy:  config.Proxy,
		Token:  token,
	}
	ytClient, err := ythttp.NewClient(ytConfig)
	if err != nil {
		return nil, err
	}
	return &YTReader{
		ytClient: ytClient,
		config:   config,
		logger:   logger,
	}, nil
}

func (r *YTReader) ReadLatestTable(ctx context.Context, consumer VariantsConsumer) error {
	if consumer == nil {
		return xerrors.New("consumer shall not be null")
	}
	p := ypath.Path(r.config.TablePath)
	ok, err := r.ytClient.NodeExists(ctx, p, nil)
	if err != nil {
		return err
	} else if !ok {
		return fmt.Errorf("YT table %v is not found", r.config.TablePath)
	}
	var tables []string
	err = r.ytClient.ListNode(ctx, p, &tables, nil)
	if err != nil {
		return err
	}
	latestTable, err := slices.SelectString(tables, slices.MaxSelectorString)
	if err != nil {
		return err
	}
	r.logger.Infof("YT table to read: %v", latestTable)
	ytReader, err := r.ytClient.ReadTable(
		ctx, ypath.Path(fmt.Sprintf("%s/%s", r.config.TablePath, latestTable)), &yt.ReadTableOptions{Unordered: true})
	if err != nil {
		return err
	}
	defer r.ytClient.Stop()

	count := 0
	for ytReader.Next() {
		count++
		var row TDVariants
		err := ytReader.Scan(&row)
		if err != nil {
			return fmt.Errorf("error while scanning the table: %+v", err)
		}
		err = consumer(ctx, row)
		if err != nil {
			r.logger.Warnf("Unable to consume the record %T %#v", err, err)
			r.logger.Warnf("Unable to consume the record %v: %+v", row.QID, err)
			continue
		}
		if count == 500 || count%5000 == 0 {
			r.logger.Infof("Processed %d records from the YT table %v", count, latestTable)
		}
		if r.lowOnMemory() {
			r.logger.Warnf(
				"skippng the rest of the table after %v records (reason: mem usage %v)",
				count,
				withSeparators(getConsumption()),
			)
			break
		}
	}
	r.logger.Infof("Processed %d records from the YT table %v", count, latestTable)

	return ytReader.Err()
}

func (r *YTReader) lowOnMemory() bool {
	if r.config.MaxMemoryConsumption <= 0 {
		return false
	}
	consumption := float64(getConsumption()) / (float64(r.config.MaxMemoryConsumption) * 1024. * 1024. * 1024.)
	return consumption >= 1
}

func getConsumption() uint64 {
	var m runtime.MemStats
	runtime.ReadMemStats(&m)
	return m.Alloc
}

func withSeparators(value uint64) string {
	msg := message.NewPrinter(language.Russian)
	return msg.Sprintf("%d", value)
}
