package consumers

import (
	"context"
	"strconv"
	"strings"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/weekendtour/internal/models"
	"a.yandex-team.ru/travel/avia/weekendtour/internal/readers/ytreader"
	"a.yandex-team.ru/travel/avia/weekendtour/internal/service/filters"
	"a.yandex-team.ru/travel/avia/weekendtour/internal/service/utils"
)

type Repository interface {
	StoreRoundTrip(context.Context, models.RoundTrip) (*models.RoundTrip, error)
	GetRoundTrips(context.Context, uint) ([]models.RoundTrip, error)
}

type Cache interface {
	CacheRoundTrip(models.RoundTrip)
}

type RoundTripFilter interface {
	FilterName() string
	AcceptsQID(string) bool
	AcceptsJourney(ytreader.JourneyBoundaries) bool
	AcceptsVariant(models.RoundTrip) bool
}

type RoundTripConsumer struct {
	repository Repository
	cache      Cache
	filters    []RoundTripFilter
	logger     log.Logger
}

func NewRoundTripConsumer(ctx context.Context, repository Repository, logger log.Logger) (*RoundTripConsumer, error) {
	minPriceFilter := filters.NewMinRoundTripPriceFilter()
	roundTrips, err := repository.GetRoundTrips(ctx, utils.GetWeekendIDForDate(time.Now()))
	if err != nil {
		return nil, err
	}
	logger.Info("Caching round-trips into memory", log.Int("count", len(roundTrips)))
	for _, roundTrip := range roundTrips {
		minPriceFilter.CacheRoundTrip(roundTrip)
	}
	return &RoundTripConsumer{
		repository: repository,
		cache:      minPriceFilter,
		filters: []RoundTripFilter{
			filters.NewRoundTripFilter(),
			minPriceFilter,
		},
		logger: logger,
	}, nil
}

func (rc *RoundTripConsumer) Consume(ctx context.Context, variants ytreader.TDVariants) error {
	for _, filter := range rc.filters {
		if !filter.AcceptsQID(variants.QID) {
			rc.logger.Debug(
				"Rejected by QID",
				log.String("qid", variants.QID),
				log.String("filter", filter.FilterName()),
			)
			return nil
		}
	}

	journey, err := variants.ParseQID()
	if err != nil {
		rc.logger.Debug("Rejected by parsing QID", log.String("qid", variants.QID))
		return err
	}

	for _, filter := range rc.filters {
		if !filter.AcceptsJourney(journey) {
			rc.logger.Debug(
				"Rejected by journey",
				log.String("qid", variants.QID),
				log.String("filter", filter.FilterName()),
			)
			return nil
		}
	}

	if variants.Variants == nil {
		rc.logger.Debug("Rejected by no variants", log.String("qid", variants.QID))
		return nil
	}

	for _, variant := range variants.Variants {
		roundTrip, err := parseVariant(journey, variant)
		if err != nil {
			rc.logger.Debug("Rejected by parsing variant", log.String("qid", variants.QID))
			return err
		}
		for _, filter := range rc.filters {
			if !filter.AcceptsVariant(roundTrip) {
				rc.logger.Debug(
					"Rejected by variant",
					log.String("qid", variants.QID),
					log.String("filter", filter.FilterName()),
				)
			}
		}
		rc.logger.Debug("Accepted", log.String("qid", variants.QID))
		_, err = rc.repository.StoreRoundTrip(ctx, roundTrip)
		if err != nil {
			rc.logger.Debug("Unable to store variant", log.String("qid", variants.QID))
			return err
		}
		rc.cache.CacheRoundTrip(roundTrip)
	}
	return nil
}

func parseVariant(journey ytreader.JourneyBoundaries, variant ytreader.Variant) (models.RoundTrip, error) {
	result := models.RoundTrip{
		PointFromKey: journey.PointFromKey,
		PointToKey:   journey.PointToKey,
		Currency:     variant.Price.Currency,
		Price:        variant.Price.Value,
	}

	// Since "RUR" and "RUB" are referring to the same currency, always use one of them
	if result.Currency == "RUR" {
		result.Currency = "RUB"
	}

	for index, route := range variant.Route {
		if index == 0 {
			result.ForwardFlights = strings.Join(route, utils.FlightFieldSeparator)
		} else {
			result.BackwardFlights = strings.Join(route, utils.FlightFieldSeparator)
		}
	}

	weekendID, err := utils.GetWeekendID(result.ForwardFlights)
	result.WeekendID = weekendID
	if err != nil {
		return result, err
	}
	return result, nil
}

func getPointID(pointCode string) (uint32, error) {
	// TODO(u-jeen): use stationID or cityID from
	result, err := strconv.Atoi(strings.TrimPrefix(strings.TrimPrefix(pointCode, "s"), "c"))
	return uint32(result), err
}
