package filters

import (
	"sync"

	"a.yandex-team.ru/travel/avia/weekendtour/internal/models"
	"a.yandex-team.ru/travel/avia/weekendtour/internal/readers/ytreader"
)

type minRoundTripPricesCacheKey struct {
	forwardFlights string
	backwardFights string
}

type minRoundTripPricesCache struct {
	data map[minRoundTripPricesCacheKey]float64
	mu   sync.Mutex
}

func NewMinRoundTripPriceFilter() *minRoundTripPricesCache {
	return &minRoundTripPricesCache{
		data: make(map[minRoundTripPricesCacheKey]float64),
	}
}

func newMinRoundTripPricesCacheKey(roundTrip models.RoundTrip) minRoundTripPricesCacheKey {
	return minRoundTripPricesCacheKey{
		forwardFlights: roundTrip.ForwardFlights,
		backwardFights: roundTrip.BackwardFlights,
	}
}

func (f *minRoundTripPricesCache) FilterName() string {
	return "minPriceCache"
}

func (f *minRoundTripPricesCache) AcceptsQID(_ string) bool {
	return true
}

func (f *minRoundTripPricesCache) AcceptsJourney(journey ytreader.JourneyBoundaries) bool {
	return true

}

func (f *minRoundTripPricesCache) AcceptsVariant(roundTrip models.RoundTrip) bool {
	cacheKey := newMinRoundTripPricesCacheKey(roundTrip)
	currentMinPrice, ok := f.data[cacheKey]
	if !ok {
		return true
	}
	return currentMinPrice >= roundTrip.Price
}

func (f *minRoundTripPricesCache) CacheRoundTrip(roundTrip models.RoundTrip) {
	f.mu.Lock()
	defer f.mu.Unlock()

	cacheKey := newMinRoundTripPricesCacheKey(roundTrip)
	currentMinPrice, ok := f.data[cacheKey]
	if !ok || currentMinPrice > roundTrip.Price {
		f.data[cacheKey] = roundTrip.Price
	}
}
