package main

import (
	"context"
	"fmt"

	"github.com/heetch/confita"
	"github.com/heetch/confita/backend/env"
	"github.com/heetch/confita/backend/flags"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/library/go/logging"
	wizardtanker "a.yandex-team.ru/travel/avia/wizard/pkg/tanker"
	"a.yandex-team.ru/travel/library/go/tanker"
)

type Config struct {
	BranchID      string `config:"branch_id,required"`
	ProjectID     string `config:"project_id,required"`
	KeysetID      string `config:"keyset_id,required"`
	Token         string `config:"TOKEN,required,backend=env"`
	InputFile     string `config:"input,required"`
	TankerBaseURL string
}

func main() {
	logger, _ := logging.NewLogger(&logging.Config{
		Level:               log.InfoLevel,
		StdoutLoggingConfig: &logging.StdoutLoggingConfig{},
	})
	config, err := getConfig()
	if err != nil {
		fmt.Println(err)
		return
	}

	tankerAPI := tanker.NewClient(config.TankerBaseURL, config.Token)
	keyset, err := wizardtanker.ReadWizardKeyset(config.InputFile, logger)
	if err != nil {
		logger.Error(err.Error())
		return
	}
	const keysetFileFormat = "tjson"
	err = tankerAPI.ReplaceKeyset(config.BranchID, config.ProjectID, config.KeysetID, keysetFileFormat, keyset)
	if err != nil {
		logger.Error(err.Error())
	}
}

func getConfig() (*Config, error) {
	loader := confita.NewLoader(env.NewBackend(), flags.NewBackend())
	config := Config{
		TankerBaseURL: "https://tanker-api.yandex-team.ru",
		BranchID:      "go-wizard",
		ProjectID:     "avia_wizard",
		KeysetID:      "wizard",
		InputFile:     "wizard_default_keyset.json",
	}
	err := loader.Load(context.Background(), &config)
	return &config, err
}
