package main

import (
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/yandex/geobase"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/flights"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/handlers"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/handlers/building"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/landings"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/mappers"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/point"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/point/finder"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/point/providers"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/search"
	resultslib "a.yandex-team.ru/travel/avia/wizard/pkg/wizard/lib/results"
	pointParse "a.yandex-team.ru/travel/avia/wizard/pkg/wizard/logging/yt/point"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/logging/yt/show"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/services/featureflag"
	flightsService "a.yandex-team.ru/travel/avia/wizard/pkg/wizard/services/flights"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/services/personalization"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/services/tdapi"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/settings"
)

func buildPointParser(
	appRepositories *applicationRepositories,
	nearestAviaSettlementProvider *providers.NearestAviaSettlementProvider,
	geobaseClient geobase.Geobase,
) *point.Parser {
	pointFinder := finder.NewFinder(
		appRepositories.nameSearchIndex,
		appRepositories.settlementRepository,
		appRepositories.stationRepository,
	)
	pointByKeyProvider := providers.NewPointByPointKey(
		appRepositories.countryRepository,
		appRepositories.settlementRepository,
		appRepositories.regionRepository,
		appRepositories.stationRepository,
	)
	geoPointProvider := providers.NewPointProviderByGeoID(
		appRepositories.countryRepository,
		appRepositories.settlementRepository,
		appRepositories.regionRepository,
		geobaseClient,
	)
	titlePointProvider := providers.NewPointByTitleProvider(pointFinder)
	return point.NewParser(
		pointByKeyProvider,
		geoPointProvider,
		titlePointProvider,
		nearestAviaSettlementProvider,
		appRepositories.settlementRepository,
		appRepositories.stationToSettlementRepository,
		appRepositories.popularSettlementsRepository,
		appRepositories.directionRepository,
	)
}

func buildRouteHandler(
	appRepositories *applicationRepositories,
	appSettings *settings.Settings,
	appLogger log.Logger,
	pointParseLogger *pointParse.Logger,
	showLogger *show.Logger,
	sharedFlightsClient flightsService.Provider,
	appTranslators *applicationTranslators,
	ticketDaemonAPIClient tdapi.Client,
	personalizationClient personalization.Client,
	flightNumberNormalizer *flights.FlightNumberNormalizer,
	geobaseClient geobase.Geobase,
	featureFlagStorage *featureflag.Storage,
) *handlers.RouteHandler {
	landingBuilder := landings.NewFrontLandingBuilder(appSettings, appRepositories.translatedTitleRepository)
	nearestAviaSettlementProvider := providers.NewNearestAviaSettlementProvider(
		appRepositories.settlementRepository,
		appSettings.NearestSettlementMaxRadiusKM,
	)
	pointParser := buildPointParser(appRepositories, nearestAviaSettlementProvider, geobaseClient)
	minPriceToSearchResultMapper := mappers.NewWizardToSearchResultMapper(
		appLogger,
		appRepositories.stationRepository,
		appRepositories.partnerRepository,
		appRepositories.aviaCompanyRepository,
		appRepositories.companyRepository,
		appRepositories.translatedTitleRepository,
		appRepositories.companyTariffRepository,
		appRepositories.locationRepository,
		appRepositories.checkedBaggageRepository,
	)

	flightBuilder := building.NewFlightResultBuilder(
		appTranslators.flight,
		appTranslators.time,
		appTranslators.common,
		appRepositories.settlementRepository,
		appRepositories.translatedTitleRepository,
		landingBuilder,
		appRepositories.stationRepository,
	)

	flightHandler := handlers.NewFlightHandler(
		appLogger,
		appRepositories.companyRepository,
		appRepositories.countryRepository,
		appRepositories.regionRepository,
		appRepositories.settlementRepository,
		appRepositories.settlementTimeZoneRepository,
		appRepositories.stationToSettlementRepository,
		landingBuilder,
		appRepositories.bannedFlightRepository,
		sharedFlightsClient,
		flightBuilder,
	)

	urlBuilder := building.NewURLBuilder()
	variantsBuilder := building.NewVariantsBuilder(
		appRepositories.settlementRepository,
		appRepositories.translatedTitleRepository,
		appTranslators.pointToPoint,
		appRepositories.stationRepository,
		appRepositories.aviaCompanyRepository,
		appRepositories.companyTariffRepository,
		urlBuilder,
		appLogger,
		showLogger,
	)
	pointToPointResponseBuilder := building.NewPointToPointResponseBuilder(
		variantsBuilder,
		appRepositories.translatedTitleRepository,
		appRepositories.companyRepository,
		appRepositories.stationRepository,
		landingBuilder,
		appTranslators.pointToPoint,
		appTranslators.common,
		nearestAviaSettlementProvider,
		appRepositories.directionRepository,
		appRepositories.partnerRepository,
		appRepositories.minPriceRepository,
		appRepositories.currencyRepository,
		appRepositories.routeCountRepository,
		appLogger,
	)
	resultsMerger := resultslib.NewMergerByConversion(&appRepositories.conversionRepository)
	resultProvider := search.NewResultProvider(
		appLogger,
		appRepositories.resultsRepository,
		ticketDaemonAPIClient,
		personalizationClient,
		resultsMerger,
		true,
		true,
		appSettings.Search.FetchVariantsFromPartnerTable,
	)
	pointToPointHandler := handlers.NewPointToPointHandler(
		appLogger,
		appRepositories.settlementRepository,
		appRepositories.regionRepository,
		pointParser,
		resultProvider,
		appRepositories.locationRepository,
		minPriceToSearchResultMapper,
		pointToPointResponseBuilder,
		appSettings.Personalization.Enabled,
		featureFlagStorage,
	)

	commonResponseBuilder := building.NewCommonResponseBuilder(
		landingBuilder,
		appTranslators.common,
		appRepositories.translatedTitleRepository,
	)
	commonHandler := handlers.NewCommonHandler(
		appLogger,
		commonResponseBuilder,
		appRepositories.locationRepository,
		appRepositories.settlementRepository,
		appRepositories.regionRepository,
	)

	return handlers.NewRouteHandler(
		appLogger,
		pointParseLogger,
		flightHandler,
		pointParser,
		pointToPointHandler,
		commonHandler,
		flightNumberNormalizer,
	)
}
