package main

import (
	"context"
	"fmt"
	"net/http"
	"net/http/pprof"

	"github.com/go-chi/chi/v5"
	"github.com/jonboulle/clockwork"
	"github.com/opentracing/opentracing-go"
	"google.golang.org/grpc"

	"a.yandex-team.ru/kikimr/public/sdk/go/ydb/table"
	"a.yandex-team.ru/library/go/core/log"
	arczap "a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/core/metrics/solomon"
	"a.yandex-team.ru/library/go/yandex/geobase"
	aviaHandlers "a.yandex-team.ru/travel/avia/library/go/handlers"
	aviaMetrics "a.yandex-team.ru/travel/avia/library/go/metrics"
	"a.yandex-team.ru/travel/avia/library/go/probes"
	"a.yandex-team.ru/travel/avia/library/go/services/backend"
	"a.yandex-team.ru/travel/avia/library/go/services/featureflag"
	wizardgeobase "a.yandex-team.ru/travel/avia/wizard/pkg/geobase"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/caches/references"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/flags"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/flights"
	domainHandlers "a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/handlers"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/handlers/building"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/landings"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/mappers"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/parameters"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/parameters/checkers"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/parameters/dynamic"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/point/providers"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/search"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/handlers"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/helpers"
	resultslib "a.yandex-team.ru/travel/avia/wizard/pkg/wizard/lib/results"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/logging"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/logging/yt/point"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/logging/yt/query"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/logging/yt/response"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/logging/yt/service"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/logging/yt/show"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/metrics"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/repositories"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/repositories/ydb"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/services/dynamicresources"
	appfeatureflag "a.yandex-team.ru/travel/avia/wizard/pkg/wizard/services/featureflag"
	flightsService "a.yandex-team.ru/travel/avia/wizard/pkg/wizard/services/flights"
	sharedFlights "a.yandex-team.ru/travel/avia/wizard/pkg/wizard/services/flights/shared"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/services/personalization"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/services/tdapi"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/settings"
	"a.yandex-team.ru/travel/library/go/services/sandbox"
)

type application struct {
	settings                  *settings.Settings
	logger                    *arczap.Logger
	responseJSONLogger        *arczap.Logger
	server                    *chi.Mux
	ytLogging                 *appLogging
	queryParametersParser     parameters.IQueryParametersParser
	ydbSessionPool            *table.SessionPool
	translators               *applicationTranslators
	repositories              *applicationRepositories
	httpClient                *http.Client
	sandboxClient             sandbox.Client
	sharedFlightsClient       flightsService.Provider
	ticketDaemonAPIClient     *tdapi.TicketDaemonAPIClient
	personalizationClient     *personalization.GrpcPersonalizationClient
	geobaseClient             geobase.Geobase
	resourceRegistry          *references.ResourceRegistry
	partnerReference          *references.Partners
	personalizationConnection grpc.ClientConnInterface
	featureFlagStorage        *appfeatureflag.Storage
}

func newApplication(settings *settings.Settings) *application {
	return &application{
		settings: settings,
	}
}

type appLogging struct {
	pointParseLogger  *point.Logger
	queryLogger       *query.Logger
	responseLogger    *response.Logger
	showLogger        *show.Logger
	serviceCallLogger *service.Logger
}

func (app *application) configureLogging() func() {
	appLogger := logging.NewApplicationLogger(app.settings.Logging)
	app.logger = appLogger
	app.logger.Info("configuring logging")
	app.responseJSONLogger = logging.NewResponseJSONLogger(app.settings.Logging)
	logPath := app.settings.Logging.LogPath
	app.ytLogging = &appLogging{
		pointParseLogger:  point.NewLogger(logPath, app.settings.Logging.Yt.PointParseLogFileName, appLogger),
		queryLogger:       query.NewLogger(logPath, app.settings.Logging.Yt.QueryLogFileName, appLogger),
		responseLogger:    response.NewLogger(logPath, app.settings.Logging.Yt.ResponseLogFileName, appLogger),
		showLogger:        show.NewLogger(logPath, app.settings.Logging.Yt.ShowLogFileName),
		serviceCallLogger: service.NewLogger(logPath, app.settings.Logging.Yt.ServiceCallLogFileName, appLogger),
	}

	return func() {
		_ = app.logger.L.Sync()
		app.ytLogging.queryLogger.Close()
		app.ytLogging.responseLogger.Close()
		app.ytLogging.pointParseLogger.Close()
		app.ytLogging.showLogger.Close()
	}
}

func (app *application) configureYDBSessionPool() func() {
	app.logger.Info("configuring ydb session pool")
	ydbCtx := context.Background()
	ydbSessionPool, err := ydb.GetSessionPool(ydbCtx, app.settings.YdbSettings)
	helpers.PanicIfNotNil(err)
	app.ydbSessionPool = ydbSessionPool
	return func() {
		_ = ydbSessionPool.Close(ydbCtx)
	}
}

func (app *application) configureQueryParameterParser() {
	aviaDynamicParser := dynamic.NewAviaDynamicParser()
	flagParser := flags.NewParser()
	app.queryParametersParser = parameters.NewQueryParametersParser(
		aviaDynamicParser,
		flagParser,
		checkers.NewWizardChecker(),
		app.repositories.companyRepository,
	)
}

func (app *application) configureServer() {
	app.server = chi.NewRouter()

	httpMetricsRegistry := solomon.NewRegistry(solomon.NewRegistryOpts())
	app.configureMetrics(httpMetricsRegistry)
	app.logger.Info("registering handlers")

	app.configureProbes()
	app.configureEmptyOrganicHandler()
	app.configureWizardHandler(httpMetricsRegistry)
	app.configureVariantsInfoHandler(httpMetricsRegistry)
	app.configurePProf()
}

func (app *application) configureMetrics(httpMetricsRegistry *solomon.Registry) {

	httpMetricsHandler := aviaHandlers.NewMetricsHandler("http", httpMetricsRegistry)
	app.server.Get(httpMetricsHandler.GetRoute(), httpMetricsHandler.ServeHTTP)

	perfMetricsHandler := aviaHandlers.NewPerfMetricsHandler(
		app.settings.Metrics.Perf.Prefix,
		app.settings.Metrics.Perf.RefreshInterval,
	)
	app.server.Get(perfMetricsHandler.GetRoute(), perfMetricsHandler.ServeHTTP)

	appMetricsRegistry := aviaMetrics.NewAppMetricsRegistry("upper")
	wizardMetrics := metrics.NewWizardMetrics(appMetricsRegistry)
	metrics.SetGlobalWizardMetrics(wizardMetrics)
	appMetricsHandler := aviaHandlers.NewAppMetricsHandler(appMetricsRegistry)
	app.server.Get(appMetricsHandler.GetRoute(), appMetricsHandler.ServeHTTP)
}

func (app *application) configureProbes() {
	probeService := probes.NewState(app.logger)
	probes.ChiBind(app.settings.Probes, app.server, probeService)
}

func (app *application) configureEmptyOrganicHandler() {
	emptyOrganicHandler := handlers.EmptyOrganicHandler{}
	app.server.Get("/organic/", emptyOrganicHandler.ServeHTTP)
	app.server.Get("/organic-common/", emptyOrganicHandler.ServeHTTP)
}

func (app *application) configureWizardHandler(httpMetricsRegistry *solomon.Registry) {
	jobIDGenerator := helpers.NewJobIDGenerator()
	wizardHandler := app.buildWizardHandler()
	app.server.Route(
		wizardHandler.GetRoute(), func(r chi.Router) {
			for _, m := range handlers.GetWizardHandlerMiddlewares(
				jobIDGenerator,
				app.logger,
				opentracing.GlobalTracer(),
				httpMetricsRegistry,
				app.settings.MaxInflightRequests,
			) {
				r.Use(m)
			}
			r.Get("/", wizardHandler.ServeHTTP)
		},
	)
}

func (app *application) configureVariantsInfoHandler(httpMetricsRegistry *solomon.Registry) {
	variantsInfoHandler := app.buildVariantsInfoHandler()
	app.server.Route(
		variantsInfoHandler.GetRoute(), func(r chi.Router) {
			for _, m := range handlers.GetVariantsInfoHandlerMiddlewares(
				app.logger,
				opentracing.GlobalTracer(),
				httpMetricsRegistry,
				app.settings.MaxVariantsInfoInflightRequests,
			) {
				r.Use(m)
			}
			r.Get("/", variantsInfoHandler.ServeHTTP)
		},
	)
}

func (app *application) configureTranslators() {
	app.logger.Info("configuring translators")
	app.translators = buildTranslators(app.settings, app.logger, app.repositories)
}

func (app *application) configureReferences() {
	app.logger.Info("configuring references")
	app.resourceRegistry = references.NewRegistry(app.settings.References.Resources, app.logger)
	if err := app.resourceRegistry.OnUpdate(""); err != nil {
		helpers.PanicIfNotNil(fmt.Errorf("failed to update dicts registry: %w", err))
	}

	backendClient := backend.NewHTTPClient(app.settings.References.BackendBaseURL, app.httpClient)
	app.partnerReference = references.NewPartners(backendClient, app.settings.References.RecachePartnersInterval, app.logger)
	if err := app.partnerReference.Precache(); err != nil {
		helpers.PanicIfNotNil(fmt.Errorf("failed to precache partner reference: %w", err))
	}
}

func (app *application) configureGeobase() func() {
	if app.settings.GeobaseConfig.Enabled {
		app.logger.Info("configuring geobase")
		geobaseClient, err := wizardgeobase.NewGeobase(app.settings.GeobaseConfig)
		helpers.PanicIfNotNil(err)
		app.geobaseClient = geobaseClient
		return geobaseClient.Destroy
	}
	return func() {}
}

func (app *application) configureRepositories() {
	app.logger.Info("configuring repositories")
	app.repositories = buildRepositories(app.resourceRegistry, app.settings, app.logger, app.ydbSessionPool, app.partnerReference)
	app.configureConversionRepository()
}

func (app *application) configureConversionRepository() {
	app.logger.Info("configuring conversion repository")

	app.repositories.conversionRepository = *repositories.NewConversionRepository(
		app.logger,
		app.settings.Conversion,
		app.repositories.partnerRepository,
		app.resourceRegistry,
	)
	helpers.PanicIfNotNil(app.repositories.conversionRepository.Precache())
}

func (app *application) configureHTTPClient() {
	app.httpClient = &http.Client{Timeout: app.settings.HTTPClientTimeout}
}

func (app *application) initPersonalizationConnection() error {
	personalizationConnection, err := personalization.CreateConnection(app.settings.Personalization, app.logger)
	if err != nil {
		return fmt.Errorf("failed to create connection to personalization service: %w", err)
	}
	app.personalizationConnection = personalizationConnection
	return nil
}

func (app *application) configureClients() error {
	app.configurSharedFlightsClient()
	app.ticketDaemonAPIClient = tdapi.NewTicketDaemonAPIClient(
		app.httpClient,
		app.settings.TicketDaemonAPIBaseURL,
		app.logger,
		app.ytLogging.serviceCallLogger,
	)
	if err := app.initPersonalizationConnection(); err != nil {
		return err
	}
	app.personalizationClient = personalization.NewPersonalizationClient(
		app.personalizationConnection,
		app.settings.Personalization.AppTimeout,
		app.logger,
		app.ytLogging.serviceCallLogger,
	)

	featureFlagClient := featureflag.NewClient(
		app.settings.FeatureFlag.URL,
		"wizard",
		app.logger,
		app.httpClient,
	)
	featureFlagStorage := featureflag.NewStorage(
		featureFlagClient,
		app.settings.FeatureFlag.UpdateInterval,
		app.logger,
		clockwork.NewRealClock(),
	)
	featureFlagStorage.StartPeriodicUpdates()
	app.featureFlagStorage = appfeatureflag.NewStorage(featureFlagStorage)

	return nil
}

func (app *application) configurSharedFlightsClient() {
	flightPointMapper := sharedFlights.NewPointMapper(
		app.repositories.stationRepository,
		app.repositories.locationRepository,
	)
	flightsMapper := sharedFlights.NewFlightsMapper(flightPointMapper, app.repositories.companyRepository)

	app.sharedFlightsClient = sharedFlights.NewSharedFlightsClient(
		app.httpClient,
		app.settings.SharedFlightsBaseURL,
		app.logger,
		app.ytLogging.serviceCallLogger,
		flightsMapper,
	)
}

func (app *application) buildWizardHandler() *handlers.WizardHandler {
	flightNumberNormalizer := flights.NewFlightNumberNormalizer(app.repositories.companyRepository)
	routeHandler := buildRouteHandler(
		app.repositories,
		app.settings,
		app.logger,
		app.ytLogging.pointParseLogger,
		app.ytLogging.showLogger,
		app.sharedFlightsClient,
		app.translators,
		app.ticketDaemonAPIClient,
		app.personalizationClient,
		flightNumberNormalizer,
		app.geobaseClient,
		app.featureFlagStorage,
	)
	domainWizardHandler := domainHandlers.NewWizardHandler(
		app.logger,
		app.ytLogging.queryLogger,
		app.ytLogging.responseLogger,
		app.queryParametersParser,
		routeHandler,
		app.settings,
		flightNumberNormalizer,
	)
	return handlers.NewWizardHandler(
		app.logger,
		app.responseJSONLogger,
		app.settings,
		domainWizardHandler,
	)
}

func (app *application) buildVariantsInfoHandler() *handlers.VariantsInfoHandler {
	urlBuilder := building.NewURLBuilder()
	variantsBuilder := building.NewVariantsBuilder(
		app.repositories.settlementRepository,
		app.repositories.translatedTitleRepository,
		app.translators.pointToPoint,
		app.repositories.stationRepository,
		app.repositories.aviaCompanyRepository,
		app.repositories.companyTariffRepository,
		urlBuilder,
		app.logger,
		app.ytLogging.showLogger,
	)
	landingBuilder := landings.NewFrontLandingBuilder(app.settings, app.repositories.translatedTitleRepository)
	respBuilder := building.NewVariantsInfoResponseBuilder(
		variantsBuilder,
		app.repositories.translatedTitleRepository,
		landingBuilder,
		app.repositories.directionRepository,
		app.repositories.companyRepository,
		app.translators.pointToPoint,
		app.logger,
	)
	nearestAviaSettlementProvider := providers.NewNearestAviaSettlementProvider(
		app.repositories.settlementRepository,
		app.settings.NearestSettlementMaxRadiusKM,
	)
	minPriceToSearchResultMapper := mappers.NewWizardToSearchResultMapper(
		app.logger,
		app.repositories.stationRepository,
		app.repositories.partnerRepository,
		app.repositories.aviaCompanyRepository,
		app.repositories.companyRepository,
		app.repositories.translatedTitleRepository,
		app.repositories.companyTariffRepository,
		app.repositories.locationRepository,
		app.repositories.checkedBaggageRepository,
	)
	resultsMerger := resultslib.NewMergerByConversion(&app.repositories.conversionRepository)
	resultProvider := search.NewResultProvider(
		app.logger,
		app.repositories.resultsRepository,
		app.ticketDaemonAPIClient,
		app.personalizationClient,
		resultsMerger,
		false,
		false,
		app.settings.Search.FetchVariantsFromPartnerTable,
	)
	handler := domainHandlers.NewVariantsInfoHandler(
		app.logger,
		buildPointParser(app.repositories, nearestAviaSettlementProvider, app.geobaseClient),
		resultProvider,
		minPriceToSearchResultMapper,
		respBuilder,
	)
	queryParametersParser := parameters.NewQueryParametersParser(
		dynamic.NewAviaDynamicParser(),
		flags.NewParser(),
		checkers.NewVariantsInfoQueryChecker(),
		app.repositories.companyRepository,
	)
	return handlers.NewVariantsInfoHandler(
		app.logger,
		app.responseJSONLogger,
		app.settings,
		handler,
		queryParametersParser,
	)
}

func (app *application) run() {
	app.logger.Info("start listening...", log.Int("port", int(app.settings.Port)))
	app.logger.Fatal(
		"server error",
		log.Error(
			http.ListenAndServe(
				fmt.Sprintf(":%d", app.settings.Port),
				app.server,
			),
		),
	)
}

func (app *application) configurePProf() {
	app.server.HandleFunc("/debug/pprof/", pprof.Index)
	app.server.HandleFunc("/debug/pprof/cmdline", pprof.Cmdline)
	app.server.HandleFunc("/debug/pprof/profile", pprof.Profile)
	app.server.HandleFunc("/debug/pprof/symbol", pprof.Symbol)
	app.server.HandleFunc("/debug/pprof/trace", pprof.Trace)
	app.server.HandleFunc("/debug/pprof/goroutine", pprof.Handler("goroutine").ServeHTTP)
	app.server.HandleFunc("/debug/pprof/heap", pprof.Handler("heap").ServeHTTP)
	app.server.HandleFunc("/debug/pprof/threadcreate", pprof.Handler("threadcreate").ServeHTTP)
	app.server.HandleFunc("/debug/pprof/block", pprof.Handler("block").ServeHTTP)
}

func (app *application) runDynamicResourcesService() {
	dynamicResourceService := dynamicresources.NewService(
		app.logger,
		app.settings.DynamicResources,
		app.resourceRegistry.OnUpdate,
	)
	dynamicResourceService.BackroundRun()
}
