package tanker

import (
	"bytes"
	"fmt"
	"text/template"

	"golang.org/x/text/feature/plural"
	"golang.org/x/text/language"
	"golang.org/x/text/message"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/models"
	"a.yandex-team.ru/travel/library/go/tanker"
)

type ITranslator interface {
	GetTranslation(key string, language models.Lang) (string, error)
	GetTemplatedTranslation(
		key string,
		language models.Lang,
		placeholderValues map[string]interface{},
	) (string, error)
	GetTemplatedTranslationWithCase(
		key string,
		language models.Lang,
		placeholderValues map[string]interface{},
		declinator PointDeclinator,
	) (string, error)
	GetTemplatedPluralizedTranslation(
		key string,
		language models.Lang,
		count int,
		placeholderValues map[string]interface{},
	) (string, error)
}

type PointDeclinator func(models.Point, ...models.GrammaticalCase) (string, error)

type Translator struct {
	languages     []string
	languageToTag map[string]language.Tag
	keyset        tanker.Keyset
	logger        log.Logger
}

func NewTranslator(languages []string, keyset tanker.Keyset, logger log.Logger) *Translator {
	languageToTag := map[string]language.Tag{}
	for _, l := range languages {
		languageToTag[l] = language.MustParse(l)
	}
	return &Translator{languages: languages, languageToTag: languageToTag, keyset: keyset, logger: logger}
}

func (translator *Translator) GetTranslation(key string, language models.Lang) (string, error) {
	keysetEntry, ok := translator.keyset[key]
	if !ok {
		return "", fmt.Errorf("there is no translation for key %s in keyset", key)
	}
	if keysetEntry.Info.IsPlural {
		err := fmt.Errorf("couldn't apply non-pluralized translation for plural key %s", key)
		translator.logger.Error(err.Error())
		return "", err
	}
	translation, ok := keysetEntry.Translations[language.String()]
	if !ok || translation.Form == "" {
		return "", fmt.Errorf("there is no translation for language %s for key %s", language, key)
	}
	return translation.Form, nil
}

func (translator *Translator) GetTemplatedTranslation(
	key string,
	language models.Lang,
	placeholderValues map[string]interface{},
) (string, error) {
	if translation, err := translator.getTemplatedTranslationKeysetEntry(key, language); err != nil {
		return "", err
	} else {
		return translator.convertTemplateToString(key, translation.Form, placeholderValues, nil)
	}
}

func (translator *Translator) GetTemplatedTranslationWithCase(
	key string,
	language models.Lang,
	placeholderValues map[string]interface{},
	declinator PointDeclinator,
) (string, error) {
	if translation, err := translator.getTemplatedTranslationKeysetEntry(key, language); err != nil {
		return "", err
	} else {
		return translator.convertTemplateToString(key, translation.Form, placeholderValues, declinator)
	}
}

func (translator *Translator) getTemplatedTranslationKeysetEntry(key string, language models.Lang) (*tanker.Translation, error) {
	keysetEntry, ok := translator.keyset[key]
	if !ok {
		err := fmt.Errorf("there is no translation for key %s in keyset", key)
		translator.logger.Error(err.Error())
		return nil, err
	}
	if keysetEntry.Info.IsPlural {
		err := fmt.Errorf("couldn't apply non-pluralized translation for plural key %s", key)
		translator.logger.Error(err.Error())
		return nil, err
	}
	translation, ok := keysetEntry.Translations[language.String()]
	if !ok || translation.Form == "" {
		err := fmt.Errorf("there is no translation for language %s for key %s", language, key)
		translator.logger.Error(err.Error())
		return nil, err
	}
	return translation, nil
}

func (translator *Translator) GetTemplatedPluralizedTranslation(
	key string,
	language models.Lang,
	count int,
	placeholderValues map[string]interface{},
) (string, error) {
	languageTag := translator.languageToTag[language.String()]
	keysetEntry, ok := translator.keyset[key]
	if !ok {
		err := fmt.Errorf("there is no translation for key %s in keyset", key)
		translator.logger.Error(err.Error())
		return "", err
	}
	if !keysetEntry.Info.IsPlural {
		err := fmt.Errorf("couldn't apply pluralized translation for non-plural key %s", key)
		translator.logger.Error(err.Error())
		return "", err
	}
	translation, ok := keysetEntry.Translations[language.String()]
	if !ok {
		err := fmt.Errorf("there is no translation for language %s for key %s", language, key)
		translator.logger.Error(err.Error())
		return "", err
	}

	pluralizationCases, err := translator.getPluralizationCases(key, translation)
	if err != nil {
		return "", err
	}
	err = message.Set(languageTag, key, plural.Selectf(1, "", pluralizationCases...))
	if err != nil {
		return "", err
	}
	translationForm := message.NewPrinter(languageTag).Sprintf(key, count)
	return translator.convertTemplateToString(key, translationForm, placeholderValues, nil)
}

func (translator *Translator) getPluralizationCases(key string, translation *tanker.Translation) (cases []interface{}, err error) {
	if translation.Form1 == "" {
		err = fmt.Errorf("there is no translation form1 for key %s", key)
		translator.logger.Error(err.Error())
		return nil, err
	}
	cases = append(cases, plural.One, translation.Form1)
	if translation.Form2 == "" {
		err = fmt.Errorf("there is no translation form2 for key %s", key)
		translator.logger.Error(err.Error())
		return nil, err
	}
	if translation.Form3 == "" {
		return append(cases, plural.Other, translation.Form2), nil
	}
	return append(cases, plural.Few, translation.Form2, plural.Many, translation.Form3, plural.Other, translation.Form3), nil
}

func (translator *Translator) convertTemplateToString(
	key string,
	translationForm string,
	placeholderValues map[string]interface{},
	declinator PointDeclinator,
) (string, error) {
	translationTemplate := template.New(key)
	if declinator != nil {
		translationTemplate = translationTemplate.Funcs(map[string]interface{}{"case": declinator})
	}
	translationTemplate, err := translationTemplate.Parse(translationForm)
	if err != nil {
		translator.logger.Error(err.Error())
		return "", err
	}

	buffer := bytes.Buffer{}
	err = translationTemplate.Execute(&buffer, placeholderValues)
	if err != nil {
		translator.logger.Error(err.Error())
		return "", err
	}
	return buffer.String(), nil
}
