package references

import (
	"fmt"
	"math"

	"github.com/golang/protobuf/proto"

	"a.yandex-team.ru/library/go/ptr"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/models"
	dictsproto "a.yandex-team.ru/travel/proto/dicts/avia"
)

type aviaCompanies []*models.AviaCompany

type AviaCompany struct {
	aviaCompanies aviaCompanies
}

func NewAviaCompany() *AviaCompany {
	return &AviaCompany{
		aviaCompanies: make(aviaCompanies, 0),
	}
}

func (reference *AviaCompany) GetAll() []*models.AviaCompany {
	return reference.aviaCompanies
}

func (reference *AviaCompany) UpdateFromSource(iterator RepositoryUpdater) error {
	newState := NewAviaCompany()
	if err := iterator.Populate(newState); err != nil {
		return err
	}
	*reference = *newState
	return nil
}

func (reference *AviaCompany) Write(b []byte) (int, error) {
	message := new(dictsproto.TAviaCompany)
	if err := proto.Unmarshal(b, message); err != nil {
		return 0, fmt.Errorf("AviaCompany:Write: %w", err)
	}
	reference.aviaCompanies = append(
		reference.aviaCompanies,
		reference.convertProtoReferenceToModel(message),
	)
	return len(b), nil
}

func (reference *AviaCompany) convertProtoReferenceToModel(proto *dictsproto.TAviaCompany) *models.AviaCompany {
	aviaCompany := &models.AviaCompany{
		RaspCompanyID: int(proto.RaspCompanyId),
		CostType:      proto.CostType,
		CarryOnLength: int(proto.CarryonLength),
		CarryOnHeight: int(proto.CarryonHeight),
		CarryOnWidth:  int(proto.CarryonWidth),
	}
	if proto.BaggageWidth != 0 || proto.BaggageHeight != 0 || proto.BaggageLength != 0 {
		aviaCompany.BaggageWidth = ptr.Int(int(proto.BaggageWidth))
		aviaCompany.BaggageLength = ptr.Int(int(proto.BaggageLength))
		aviaCompany.BaggageHeight = ptr.Int(int(proto.BaggageHeight))
	}
	aviaCompany.CarryOnDimensionsSum = calculateCarryOnDimensionsSum(
		aviaCompany.CarryOnWidth,
		aviaCompany.CarryOnHeight,
		aviaCompany.CarryOnLength,
		int(proto.CarryonDimensionsSum),
	)
	aviaCompany.BaggageDimensionsSum = calculateBaggageDimensionsSum(
		aviaCompany.BaggageWidth,
		aviaCompany.BaggageHeight,
		aviaCompany.BaggageLength,
		int(proto.BaggageDimensionsSum),
		int(proto.CarryonDimensionsSum),
	)
	return aviaCompany
}

func calculateCarryOnDimensionsSum(width, height, length, carryOnDimensionSum int) int {
	if width == 0 || height == 0 || length == 0 {
		return carryOnDimensionSum
	}
	sum := width + length + height
	if carryOnDimensionSum == 0 {
		return sum
	}
	return int(math.Min(float64(sum), float64(carryOnDimensionSum)))
}

func calculateBaggageDimensionsSum(width, height, length *int, baggageDimensionsSum, carryOnDimensionSum int) int {
	if width == nil || height == nil || length == nil {
		return carryOnDimensionSum
	}
	sum := *width + *length + *height
	if baggageDimensionsSum == 0 {
		return sum
	}
	return int(math.Min(float64(sum), float64(baggageDimensionsSum)))
}
