package references

import (
	"fmt"

	"github.com/golang/protobuf/proto"

	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/models"
	dictsproto "a.yandex-team.ru/travel/proto/avia/wizard"
)

type directionKey struct {
	DepartureSettlementID int
	ArrivalSettlementID   int
	NationalVersion       string
	CurrencyID            int
}

type directionPrices map[directionKey][]models.ForwardDayPrice

func (m directionPrices) appendOnKey(key directionKey, value models.ForwardDayPrice) {
	data := m[key]
	m[key] = append(data, value)
}

type MinPrice struct {
	directionPrices directionPrices
}

func NewMinPrice() *MinPrice {
	return &MinPrice{make(directionPrices)}
}

func (ref *MinPrice) GetPricesByDirection(
	departureID,
	arrivalID,
	currencyID int,
	nationalVersion string,
) []models.ForwardDayPrice {
	key := directionKey{
		departureID,
		arrivalID,
		nationalVersion,
		currencyID,
	}
	if data, ok := ref.directionPrices[key]; ok {
		return data
	}
	return nil
}

func (ref *MinPrice) UpdateFromSource(iterator RepositoryUpdater) error {
	newState := NewMinPrice()
	if err := iterator.Populate(newState); err != nil {
		return err
	}
	*ref = *newState
	return nil
}

func (ref *MinPrice) Write(b []byte) (int, error) {
	message := new(dictsproto.TMinPrice)
	if err := proto.Unmarshal(b, message); err != nil {
		return 0, fmt.Errorf("MinPrice:Write: %w", err)
	}
	ref.directionPrices.appendOnKey(
		ref.getKey(message),
		ref.getPrice(message),
	)
	return len(b), nil
}

func (ref *MinPrice) getKey(proto *dictsproto.TMinPrice) directionKey {
	return directionKey{
		DepartureSettlementID: int(proto.DepartureSettlementID),
		ArrivalSettlementID:   int(proto.ArrivalSettlementID),
		NationalVersion:       proto.NationalVersion,
		CurrencyID:            int(proto.CurrencyID),
	}
}

func (ref *MinPrice) getPrice(proto *dictsproto.TMinPrice) models.ForwardDayPrice {
	return models.ForwardDayPrice{
		DateForward: proto.DateForward,
		Price:       proto.Price,
	}
}
