package references

import (
	"fmt"
	"sync/atomic"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/library/go/services/backend"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/models"
)

type Partners struct {
	backendClient   backend.Client
	partners        atomic.Value
	recacheInterval time.Duration
	appLogger       log.Logger
	onFetch         func()
}

func NewPartners(backendClient backend.Client, recacheInterval time.Duration, appLogger log.Logger) *Partners {
	return &Partners{backendClient: backendClient, recacheInterval: recacheInterval, appLogger: appLogger}
}

func (partnersCache *Partners) Precache() error {
	err := partnersCache.precache()
	if err != nil {
		return fmt.Errorf("couldn't precache partners: %w", err)
	}
	go partnersCache.repeatedPrecache()
	return nil
}

func (partnersCache *Partners) GetAll() []*models.Partner {
	return partnersCache.partners.Load().([]*models.Partner)
}

func (partnersCache *Partners) precache() error {
	partnersCache.appLogger.Info("Start updating partners")

	wizardPartners, err := partnersCache.backendClient.GetPartners()
	if partnersCache.onFetch != nil {
		partnersCache.onFetch()
	}
	if err != nil {
		return fmt.Errorf("an error occured while getting partners: %w", err)
	}
	partners := make([]*models.Partner, 0, len(wizardPartners))
	for _, p := range wizardPartners {
		partner := models.Partner(*p)
		partners = append(partners, &partner)
	}
	partnersCache.partners.Store(partners)
	partnersCache.appLogger.Info(fmt.Sprintf("Got %d partners from avia-backend", len(partners)))
	return nil
}

func (partnersCache *Partners) repeatedPrecache() {
	for {
		time.Sleep(partnersCache.recacheInterval)
		if err := partnersCache.precache(); err != nil {
			partnersCache.appLogger.Error("couldn't precache partners", log.Error(err))
		}
	}
}
