package references

import (
	"fmt"

	"github.com/golang/protobuf/proto"

	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/models"
	dictsproto "a.yandex-team.ru/travel/proto/dicts/avia"
)

type pointSynonymsByObjectID map[int][]*models.PointSynonym

func (mapper pointSynonymsByObjectID) ensureKey(key int) {
	if _, found := mapper[key]; !found {
		mapper[key] = make([]*models.PointSynonym, 0)
	}
}

func (mapper pointSynonymsByObjectID) add(synonym *models.PointSynonym) {
	mapper.ensureKey(synonym.ObjectID)
	slice := mapper[synonym.ObjectID]
	mapper[synonym.ObjectID] = append(slice, synonym)
}

type PointSynonym struct {
	pointSynonymsByObjectID pointSynonymsByObjectID
}

func NewPointSynonym() *PointSynonym {
	return &PointSynonym{
		pointSynonymsByObjectID: make(pointSynonymsByObjectID),
	}
}

func (ref *PointSynonym) GetByObjectID(id int) ([]*models.PointSynonym, bool) {
	synonyms, found := ref.pointSynonymsByObjectID[id]
	return synonyms, found
}

func (ref *PointSynonym) UpdateFromSource(iterator RepositoryUpdater) error {
	newState := NewPointSynonym()
	if err := iterator.Populate(newState); err != nil {
		return err
	}
	*ref = *newState
	return nil
}

func (ref *PointSynonym) Write(b []byte) (int, error) {
	message := new(dictsproto.TPointSynonym)
	if err := proto.Unmarshal(b, message); err != nil {
		return 0, fmt.Errorf("PointSynonym:Write: %w", err)
	}
	model := ref.convertProtoReferenceToModel(message)
	ref.pointSynonymsByObjectID.ensureKey(model.ObjectID)
	ref.pointSynonymsByObjectID.add(model)
	return len(b), nil
}

func (ref *PointSynonym) convertProtoReferenceToModel(proto *dictsproto.TPointSynonym) *models.PointSynonym {
	return &models.PointSynonym{
		ID:            int(proto.ID),
		Title:         proto.Title,
		ContentTypeID: int(proto.ContentTypeID),
		ObjectID:      int(proto.ObjectID),
		SearchType:    models.SearchType(proto.SearchType),
	}
}
