package references

import (
	"fmt"
	"io"
	"path/filepath"
	"strings"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/library/go/dicts/base"
)

type Config struct {
	ResourcesPath            string `config:"REFERENCE_RESOURCES_PATH" yaml:"path"`
	ResourcesConversionsPath string `config:"REFERENCE_RESOURCES_CONVERSIONS_PATH" yaml:"path_conversions"`
	UseDynamicResources      bool   `yaml:"use_dynamic_resources"`
}

var DefaultConfig = Config{
	ResourcesPath:            "/app/data",
	ResourcesConversionsPath: "/app/data",
	UseDynamicResources:      false,
}

type ResourceRegistry struct {
	writerByPath map[string]UpdatableWriter
	config       Config
	logger       log.Logger

	AviaCompany         *AviaCompany
	CityMajority        *CityMajority
	CodeSystem          *CodeSystem
	Company             *Company
	CompanyTariff       *CompanyTariff
	Country             *Country
	Currency            *Currency
	IataCorrection      *IataCorrection
	PointSynonym        *PointSynonym
	Region              *Region
	SettlementBigImage  *SettlementBigImage
	Settlements         *Settlement
	Station             *Station
	StationCode         *StationCode
	StationToSettlement *StationToSettlement
	TranslatedTitle     *TranslatedTitle

	Direction              *Direction
	MinPrice               *MinPrice
	RouteCount             *RouteCount
	SettlementPopularities *SettlementPopularities
	Conversion             *Conversion
}

type RepositoryUpdater interface {
	Populate(writer io.Writer) error
}

type UpdatableWriter interface {
	io.Writer
	UpdateFromSource(RepositoryUpdater) error
}

func populateMany(path2writer map[string]UpdatableWriter) error {
	for path, writer := range path2writer {
		bytesIterator, err := base.BuildIteratorFromFile(path)
		if err != nil {
			return fmt.Errorf("pkg.dicts.registry:populateMany %w", err)
		}
		err = bytesIterator.Populate(writer)
		if err != nil {
			return fmt.Errorf("pkg.dicts.registry:populateMany %w", err)
		}
	}
	return nil
}

func NewRegistry(c Config, logger log.Logger) *ResourceRegistry {
	return &ResourceRegistry{
		config:                 c,
		logger:                 logger,
		AviaCompany:            NewAviaCompany(),
		CityMajority:           NewCityMajority(),
		CodeSystem:             NewCodeSystem(),
		Company:                NewCompany(),
		CompanyTariff:          NewCompanyTariff(),
		Country:                NewCountry(),
		Currency:               NewCurrency(),
		IataCorrection:         NewIataCorrection(),
		PointSynonym:           NewPointSynonym(),
		Region:                 NewRegion(),
		SettlementBigImage:     NewSettlementBigImage(),
		Settlements:            NewSettlements(),
		Station:                NewStation(),
		StationCode:            NewStationCode(),
		StationToSettlement:    NewStationToSettlement(),
		TranslatedTitle:        NewTranslatedTitle(),
		Direction:              NewDirection(),
		MinPrice:               NewMinPrice(),
		RouteCount:             NewRouteCount(),
		SettlementPopularities: NewSettlementPopularities(),
		Conversion:             NewConversion(),
	}
}

func (r *ResourceRegistry) OnUpdate(source string) error {
	resourcePath, resourceConversionsPath, err := getResourcePaths(r.config)
	if err != nil {
		return fmt.Errorf("pkg.dicts.registry:OnUpdate %w", err)
	}

	err = r.onUpdateDict(source, resourcePath)
	if err != nil {
		return err
	}
	err = r.onUpdateConversion(resourceConversionsPath)
	if err != nil {
		return err
	}
	return nil
}

func (r *ResourceRegistry) onUpdateConversion(resourceConversionsPath string) error {
	err := r.Conversion.UpdateFromFile(resourceConversionsPath)
	if err != nil {
		return err
	}
	return nil
}

func (r *ResourceRegistry) onUpdateDict(source string, resourcePath string) error {
	r.updateWriterByPath(resourcePath)

	for path, writer := range r.writerByPath {
		if source == "" || strings.Contains(path, source) {
			err := r.onUpdate(path, writer)
			if err != nil {
				return err
			}
		}
	}
	return nil
}

func (r *ResourceRegistry) onUpdate(path string, writer UpdatableWriter) error {
	r.logger.Info(fmt.Sprintf("start updating resource by path %s", path))
	bytesIterator, err := base.BuildIteratorFromFile(path)
	if err != nil {
		return fmt.Errorf("pkg.dicts.registry:OnUpdate %w", err)
	}
	if err := writer.UpdateFromSource(bytesIterator); err != nil {
		return fmt.Errorf("pkg.dicts.registry:OnUpdate %w", err)
	}
	return nil
}

func (r *ResourceRegistry) updateWriterByPath(resourcePath string) {
	r.writerByPath = map[string]UpdatableWriter{
		resourcePath + "/avia_company.data":          r.AviaCompany,
		resourcePath + "/city_majority.data":         r.CityMajority,
		resourcePath + "/code_system.data":           r.CodeSystem,
		resourcePath + "/company.data":               r.Company,
		resourcePath + "/company_tariff.data":        r.CompanyTariff,
		resourcePath + "/country.data":               r.Country,
		resourcePath + "/currency.data":              r.Currency,
		resourcePath + "/iata_correction.data":       r.IataCorrection,
		resourcePath + "/point_synonym.data":         r.PointSynonym,
		resourcePath + "/region.data":                r.Region,
		resourcePath + "/settlement_big_image.data":  r.SettlementBigImage,
		resourcePath + "/settlements.data":           r.Settlements,
		resourcePath + "/station.data":               r.Station,
		resourcePath + "/station_code.data":          r.StationCode,
		resourcePath + "/station_to_settlement.data": r.StationToSettlement,
		resourcePath + "/translated_title.data":      r.TranslatedTitle,
		resourcePath + "/direction_national.data":    r.Direction,
		resourcePath + "/min_price.data":             r.MinPrice,
		resourcePath + "/route_count.data":           r.RouteCount,
		resourcePath + "/settlement_national.data":   r.SettlementPopularities,
	}
}

func getResourcePaths(config Config) (string, string, error) {
	resourcePath, err := getResourcePath(config.ResourcesPath, config.UseDynamicResources)
	if err != nil {
		return "", "", err
	}

	resourceConversionPath, err := getResourcePath(config.ResourcesConversionsPath, config.UseDynamicResources)
	if err != nil {
		return resourcePath, "", err
	}

	return resourcePath, resourceConversionPath, nil
}

func getResourcePath(resourcePath string, useDynamicResources bool) (string, error) {
	if useDynamicResources {
		symlinkTarget, err := filepath.EvalSymlinks(resourcePath)
		if err != nil {
			return "", err
		}
		resourcePath = filepath.Dir(symlinkTarget)
	}
	return resourcePath, nil
}
