package references

import (
	"fmt"

	"github.com/golang/protobuf/proto"

	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/models"
	dictsproto "a.yandex-team.ru/travel/proto/dicts/avia"
)

type idToSettlement map[int]*models.Settlement
type geoIDToSettlement map[int]*models.Settlement
type iataToSettlement map[string]*models.Settlement

type Settlement struct {
	idToSettlement    idToSettlement
	geoIDToSettlement geoIDToSettlement
	iataToSettlement  iataToSettlement
}

func NewSettlements() *Settlement {
	return &Settlement{
		idToSettlement:    make(idToSettlement),
		geoIDToSettlement: make(geoIDToSettlement),
		iataToSettlement:  make(iataToSettlement),
	}
}

func (ref *Settlement) GetAll() []*models.Settlement {
	objects := make([]*models.Settlement, 0, len(ref.idToSettlement))
	for _, value := range ref.idToSettlement {
		objects = append(objects, value)
	}
	return objects
}

func (ref *Settlement) GetByID(id int) (*models.Settlement, bool) {
	s, ok := ref.idToSettlement[id]
	return s, ok
}

func (ref *Settlement) GetByGeoID(id int) (*models.Settlement, bool) {
	s, ok := ref.geoIDToSettlement[id]
	return s, ok
}

func (ref *Settlement) GetByIataCode(iata string) (*models.Settlement, bool) {
	s, ok := ref.iataToSettlement[iata]
	return s, ok
}

func (ref *Settlement) UpdateFromSource(iterator RepositoryUpdater) error {
	newState := NewSettlements()
	if err := iterator.Populate(newState); err != nil {
		return err
	}
	*ref = *newState
	return nil
}

func (ref *Settlement) Write(b []byte) (int, error) {
	message := new(dictsproto.TSettlement)
	if err := proto.Unmarshal(b, message); err != nil {
		return 0, fmt.Errorf("Settlement:Write: %w", err)
	}
	model := ref.convertProtoReferenceToModel(message)
	ref.idToSettlement[model.ID] = &model
	ref.geoIDToSettlement[model.GeoID] = &model
	if model.IataCode != "" {
		ref.iataToSettlement[model.IataCode] = &model
	}
	return len(b), nil
}

func (ref *Settlement) convertProtoReferenceToModel(proto *dictsproto.TSettlement) models.Settlement {
	return models.Settlement{
		ID:                      int(proto.Id),
		GeoID:                   int(proto.GeoId),
		CountryID:               int(proto.CountryId),
		RegionID:                int(proto.RegionId),
		IataCode:                proto.IataCode,
		SirenaID:                proto.SirenaID,
		Address:                 proto.Address,
		Title:                   proto.Title,
		Longitude:               proto.Longitude,
		Latitude:                proto.Latitude,
		MajorityID:              int(proto.MajorityID),
		NewLTitleID:             int(proto.NewLTitleID),
		Hidden:                  int(proto.Hidden) > 0,
		TimeZone:                proto.TimeZone,
		TitleRuPrepositionVVoNa: proto.TitleRuPrepositionVVoNa,
	}
}
