package context

import (
	"fmt"
	"regexp"
	"strconv"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/lib/conversions"
)

var (
	restrictedKeySymbolsRegexp   = regexp.MustCompile("[@\t\n;,=: ]")
	restrictedValueSymbolsRegexp = regexp.MustCompile("[@\t\n;,]")
)

type PropsStore struct {
	mapper map[string]interface{}
	logger log.Logger
}

func NewPropsStore(mapper map[string]interface{}, logger log.Logger) *PropsStore {
	return &PropsStore{
		mapper: mapper,
		logger: logger,
	}
}

func (store PropsStore) Set(key string, value interface{}) {
	store.mapper[key] = value
}

func (store PropsStore) Get(key string) (interface{}, bool) {
	value, ok := store.mapper[key]
	return value, ok
}

func (store PropsStore) Contains(key string) bool {
	_, contains := store.mapper[key]
	return contains
}

func (store PropsStore) GetAll() map[string]interface{} {
	return store.mapper
}

type SearchPropsStore struct {
	PropsStore
}

func (store SearchPropsStore) Set(key string, value interface{}) {
	switch v := value.(type) {
	case bool:
		value = strconv.Itoa(conversions.BoolToInt(v))
	}
	key, err := store.normalizeKey(key)
	if err != nil {
		return
	}
	if value, err := store.normalizeValue(key, value); err == nil {
		store.PropsStore.Set(key, value)
	}
}

func (store SearchPropsStore) normalizeKey(key string) (string, error) {
	if restrictedKeySymbolsRegexp.FindStringIndex(key) != nil {
		store.logger.Warn(fmt.Sprintf("key %s contains restricted symbols", key))
		key = restrictedKeySymbolsRegexp.ReplaceAllLiteralString(key, "")
	}

	if len(key) == 0 {
		err := fmt.Errorf("key %s is empty", key)
		store.logger.Error(err.Error())
		return "", err
	}
	return key, nil
}

func (store SearchPropsStore) normalizeValue(key string, rawValue interface{}) (interface{}, error) {
	switch value := rawValue.(type) {
	case int, int8, int16, int32, int64, uint, uint8, uint16, uint32, uint64:
		return value, nil
	case string:
		if restrictedValueSymbolsRegexp.FindStringIndex(value) != nil {
			store.logger.Warn(fmt.Sprintf("value of prop %s contains restricted symbols: %s", key, value))
			value = restrictedValueSymbolsRegexp.ReplaceAllLiteralString(value, "")
		}
		return value, nil
	case []byte:
		if restrictedValueSymbolsRegexp.FindIndex(value) != nil {
			store.logger.Warn(fmt.Sprintf(
				"value of prop %s contains restricted symbols: %s", key, string(value)))
			value = restrictedValueSymbolsRegexp.ReplaceAllLiteral(value, []byte(""))
		}
		return value, nil
	}
	err := fmt.Errorf("value of prop %s is not string or number: %+v", key, rawValue)
	store.logger.Error(err.Error())
	return nil, err
}
